/*!
 * @quasar/quasar-ui-qcalendar v3.4.0
 * (c) 2021 Jeff Galbraith <jeff@quasar.dev>
 * Released under the MIT License.
 */

(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports, require('vue'), require('quasar')) :
  typeof define === 'function' && define.amd ? define(['exports', 'vue', 'quasar'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.QCalendar = {}, global.Vue, global.Quasar));
}(this, (function (exports, Vue, quasar) { 'use strict';

  function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

  var Vue__default = /*#__PURE__*/_interopDefaultLegacy(Vue);

  var version = "3.4.0";

  var PARSE_REGEX = /^(\d{4})-(\d{1,2})(-(\d{1,2}))?([^\d]+(\d{1,2}))?(:(\d{1,2}))?(:(\d{1,2}))?(.(\d{1,3}))?$/;
  var PARSE_TIME = /(\d\d?)(:(\d\d?)|)(:(\d\d?)|)/;

  var DAYS_IN_MONTH = [0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
  var DAYS_IN_MONTH_LEAP = [0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
  var DAYS_IN_MONTH_MIN = 28;
  var DAYS_IN_MONTH_MAX = 31;
  var MONTH_MAX = 12;
  var MONTH_MIN = 1;
  var DAY_MIN = 1;
  var DAYS_IN_WEEK = 7;
  var MINUTES_IN_HOUR = 60;
  var HOURS_IN_DAY = 24;
  var FIRST_HOUR = 0;
  var MILLISECONDS_IN_MINUTE = 60000;
  var MILLISECONDS_IN_HOUR = 3600000;
  var MILLISECONDS_IN_DAY = 86400000;
  var MILLISECONDS_IN_WEEK = 604800000;

  /* eslint-disable no-multi-spaces */
  var Timestamp = {
    date: '',       // YYYY-MM-DD
    time: '',       // HH:MM (optional)
    year: 0,        // YYYY
    month: 0,       // MM (Jan = 1, etc)
    day: 0,         // day of the month
    weekday: 0,     // week day
    hour: 0,        // 24-hr
    minute: 0,      // mm
    doy: 0,         // day of year
    workweek: 0,    // workweek number
    hasDay: false,  // if this timestamp is supposed to have a date
    hasTime: false, // if this timestamp is supposed to have a time
    past: false,    // if timestamp is in the past (based on `now` property)
    current: false, // if timestamp is current date (based on `now` property)
    future: false,  // if timestamp is in the future (based on `now` property)
    disabled: false // if timestamp is disabled
  };

  var TimeObject = {
    hour: 0,  // Number
    minute: 0 // Number
  };
  /* eslint-enable no-multi-spaces */

  // returns YYYY-MM-dd format
  function today () {
    var d = new Date(),
      month = '' + (d.getMonth() + 1),
      day = '' + d.getDate(),
      year = d.getFullYear();

    return [year, padNumber(month, 2), padNumber(day, 2)].join('-')
  }

  // get the start of the week (based on weekdays)
  function getStartOfWeek (timestamp, weekdays, today) {
    var start = copyTimestamp(timestamp);
    if (start.day === 1 || start.weekday === 0) {
      while (!weekdays.includes(start.weekday)) {
        start = nextDay(start);
      }
    }
    start = findWeekday(start, weekdays[0], prevDay);
    start = updateFormatted(start);
    if (today) {
      start = updateRelative(start, today, start.hasTime);
    }
    return start
  }

  // get the end of the week (based on weekdays)
  function getEndOfWeek (timestamp, weekdays, today) {
    var end = copyTimestamp(timestamp);
    // is last day of month?
    var lastDay = daysInMonth(end.year, end.month);
    if (lastDay === end.day || end.weekday === 6) {
      while (!weekdays.includes(end.weekday)) {
        end = prevDay(end);
      }
    }
    end = findWeekday(end, weekdays[weekdays.length - 1], nextDay);
    end = updateFormatted(end);
    if (today) {
      end = updateRelative(end, today, end.hasTime);
    }
    return end
  }

  // get start of the month
  function getStartOfMonth (timestamp) {
    var start = copyTimestamp(timestamp);
    start.day = DAY_MIN;
    updateFormatted(start);
    return start
  }

  // get end of the month
  function getEndOfMonth (timestamp) {
    var end = copyTimestamp(timestamp);
    end.day = daysInMonth(end.year, end.month);
    updateFormatted(end);
    return end
  }

  // returns minutes since midnight
  function parseTime (input) {
    var type = Object.prototype.toString.call(input);
    switch (type) {
      case '[object Number]':
        // when a number is given, it's minutes since 12:00am
        return input
      case '[object String]':
      {
        // when a string is given, it's a hh:mm:ss format where seconds are optional, but not used
        var parts = PARSE_TIME.exec(input);
        if (!parts) {
          return false
        }
        return parseInt(parts[1], 10) * 60 + parseInt(parts[3] || 0, 10)
      }
      case '[object Object]':
        // when an object is given, it must have hour and minute
        if (typeof input.hour !== 'number' || typeof input.minute !== 'number') {
          return false
        }
        return input.hour * 60 + input.minute
    }

    return false
  }

  // validate a timestamp
  function validateTimestamp (input) {
    return !!PARSE_REGEX.exec(input)
  }

  function compareTimestamps (ts1, ts2) {
    return JSON.stringify(ts1) === JSON.stringify(ts2)
  }

  function compareDate (ts1, ts2) {
    return getDate(ts1) === getDate(ts2)
  }

  function compareTime (ts1, ts2) {
    return getTime(ts1) === getTime(ts2)
  }

  function compareDateTime (ts1, ts2) {
    return getDateTime(ts1) === getDateTime(ts2)
  }

  // low-level parser (fast) for YYYY-MM-DD hh:mm:ss, use 'parseTimestamp' for formatted and relative updates
  function parsed (input) {
    // YYYY-MM-DD hh:mm:ss
    var parts = PARSE_REGEX.exec(input);

    if (!parts) { return null }

    return {
      date: input,
      time: padNumber(parseInt(parts[6], 10) || 0, 2) + ':' + padNumber(parseInt(parts[8], 10) || 0, 2),
      year: parseInt(parts[1], 10),
      month: parseInt(parts[2], 10),
      day: parseInt(parts[4], 10) || 1,
      hour: parseInt(parts[6], 10) || 0,
      minute: parseInt(parts[8], 10) || 0,
      weekday: 0,
      doy: 0,
      workweek: 0,
      hasDay: !!parts[4],
      hasTime: !!(parts[6] && parts[8]),
      past: false,
      current: false,
      future: false,
      disabled: false
    }
  }

  // high-level parser (slower) for YYYY-MM-DD hh:mm:ss
  function parseTimestamp (input, now) {
    var timestamp = parsed(input);
    if (timestamp === null) { return null }

    timestamp = updateFormatted(timestamp);

    if (now) {
      updateRelative(timestamp, now, timestamp.hasTime);
    }

    return timestamp
  }

  // parse from JavaScript Date
  function parseDate (date) {
    return updateFormatted({
      date: padNumber(date.getFullYear(), 4) + '-' + padNumber(date.getMonth() + 1, 2) + '-' + padNumber(date.getDate(), 2),
      time: padNumber(date.getHours() || 0, 2) + ':' + padNumber(date.getMinutes() || 0, 2),
      year: date.getFullYear(),
      month: date.getMonth() + 1,
      day: date.getDate(),
      hour: date.getHours(),
      minute: date.getMinutes(),
      weekday: 0,
      doy: 0,
      workweek: 0,
      hasDay: true,
      hasTime: !!(date.getHours() !== 0 && date.getMinutes() !== 0),
      past: false,
      current: false,
      future: false,
      disabled: false
    })
  }

  function getDayIdentifier (timestamp) {
    return timestamp.year * 100000000 + timestamp.month * 1000000 + timestamp.day * 10000
  }

  function getTimeIdentifier (timestamp) {
    return timestamp.hour * 100 + timestamp.minute
  }

  function getDayTimeIdentifier (timestamp) {
    return getDayIdentifier(timestamp) + getTimeIdentifier(timestamp)
  }

  function diffTimestamp (ts1, ts2, strict) {
    var utc1 = Date.UTC(ts1.year, ts1.month - 1, ts1.day, ts1.hour, ts1.minute);
    var utc2 = Date.UTC(ts2.year, ts2.month - 1, ts2.day, ts2.hour, ts2.minute);
    if (strict === true && utc2 < utc1) {
      // Not negative number
      // utc2 - utc1 < 0  -> utc2 < utc1 ->   NO: utc1 >= utc2
      return 0
    }
    return utc2 - utc1
  }

  // update the relative parts of a timestamp (past, current and future) based on 'now'
  function updateRelative (timestamp, now, time) {
    if ( time === void 0 ) time = false;

    var a = getDayIdentifier(now);
    var b = getDayIdentifier(timestamp);
    var current = a === b;

    if (timestamp.hasTime && time && current) {
      a = getTimeIdentifier(now);
      b = getTimeIdentifier(timestamp);
      current = a === b;
    }

    timestamp.past = b < a;
    timestamp.current = current;
    timestamp.future = b > a;

    return timestamp
  }

  function updateMinutes (timestamp, minutes, now) {
    timestamp.hasTime = true;
    timestamp.hour = Math.floor(minutes / MINUTES_IN_HOUR);
    timestamp.minute = minutes % MINUTES_IN_HOUR;
    timestamp.time = getTime(timestamp);
    if (now) {
      updateRelative(timestamp, now, true);
    }

    return timestamp
  }

  // update weekday
  function updateWeekday (timestamp) {
    timestamp.weekday = getWeekday(timestamp);

    return timestamp
  }

  // update day of year
  function updateDayOfYear (timestamp) {
    timestamp.doy = getDayOfYear(timestamp);

    return timestamp
  }

  // update workweek
  function updateWorkWeek (timestamp) {
    timestamp.workweek = getWorkWeek(timestamp);

    return timestamp
  }

  // update timestamp if disabled
  function updateDisabled (timestamp, disabledBefore, disabledAfter, disabledWeekdays, disabledDays) {
    var t = getDayIdentifier(timestamp);

    if (disabledBefore !== undefined) {
      var before = getDayIdentifier(parsed(disabledBefore));
      if (t <= before) {
        timestamp.disabled = true;
      }
    }

    if (timestamp.disabled !== true && disabledAfter !== undefined) {
      var after = getDayIdentifier(parsed(disabledAfter));
      if (t >= after) {
        timestamp.disabled = true;
      }
    }

    if (timestamp.disabled !== true && Array.isArray(disabledWeekdays) && disabledWeekdays.length > 0) {
      for (var weekday in disabledWeekdays) {
        if (disabledWeekdays[weekday] === timestamp.weekday) {
          timestamp.disabled = true;
          break
        }
      }
    }

    if (timestamp.disabled !== true && Array.isArray(disabledDays) && disabledDays.length > 0) {
      for (var day in disabledDays) {
        if (Array.isArray(disabledDays[day]) && disabledDays[day].length === 2) {
          var start = parsed(disabledDays[day][0]);
          var end = parsed(disabledDays[day][1]);
          if (isBetweenDates(timestamp, start, end)) {
            timestamp.disabled = true;
            break
          }
        }
        else {
          var d = getDayIdentifier(parseTimestamp(disabledDays[day] + ' 00:00'));
          if (d === t) {
            timestamp.disabled = true;
            break
          }
        }
      }
    }

    return timestamp
  }

  // update formatted (time string, date string, weekday, day of year and workweek)
  function updateFormatted (timestamp) {
    timestamp.hasTime = !(timestamp.hour === 0 && timestamp.minute === 0);
    timestamp.time = getTime(timestamp);
    timestamp.date = getDate(timestamp);
    timestamp.weekday = getWeekday(timestamp);
    timestamp.doy = getDayOfYear(timestamp);
    timestamp.workweek = getWorkWeek(timestamp);

    return timestamp
  }

  // get day of year
  function getDayOfYear (timestamp) {
    if (timestamp.year === 0) { return }
    return (Date.UTC(timestamp.year, timestamp.month - 1, timestamp.day) - Date.UTC(timestamp.year, 0, 0)) / 24 / 60 / 60 / 1000
  }

  // get workweek
  function getWorkWeek (timestamp) {
    var date;
    if (timestamp.year === 0) {
      date = Date.UTC(new Date());
    }
    else {
      date = makeDate(timestamp);
    }

    if (isNaN(date)) { return 0 }

    // Remove time components of date
    var weekday = new Date(date.getFullYear(), date.getMonth(), date.getDate());

    // Change date to Thursday same week
    weekday.setDate(weekday.getDate() - ((weekday.getDay() + 6) % 7) + 3);

    // Take January 4th as it is always in week 1 (see ISO 8601)
    var firstThursday = new Date(weekday.getFullYear(), 0, 4);

    // Change date to Thursday same week
    firstThursday.setDate(firstThursday.getDate() - ((firstThursday.getDay() + 6) % 7) + 3);

    // Check if daylight-saving-time-switch occurred and correct for it
    var ds = weekday.getTimezoneOffset() - firstThursday.getTimezoneOffset();
    weekday.setHours(weekday.getHours() - ds);

    // Number of weeks between target Thursday and first Thursday
    var weekDiff = (weekday - firstThursday) / (MILLISECONDS_IN_WEEK);
    return 1 + Math.floor(weekDiff)
  }

  // get weekday
  function getWeekday (timestamp) {
    var weekday = timestamp.weekday;
    if (timestamp.hasDay) {
      var floor = Math.floor;
      var day = timestamp.day;
      var month = ((timestamp.month + 9) % MONTH_MAX) + 1;
      var century = floor(timestamp.year / 100);
      var year = (timestamp.year % 100) - (timestamp.month <= 2 ? 1 : 0);

      weekday = (((day + floor(2.6 * month - 0.2) - 2 * century + year + floor(year / 4) + floor(century / 4)) % 7) + 7) % 7;
    }

    return weekday
  }

  // check if leap year
  function isLeapYear (year) {
    return ((year % 4 === 0) ^ (year % 100 === 0) ^ (year % 400 === 0)) === 1
  }

  // get days in month
  function daysInMonth (year, month) {
    return isLeapYear(year) ? DAYS_IN_MONTH_LEAP[month] : DAYS_IN_MONTH[month]
  }

  // copy timestamp
  function copyTimestamp (timestamp) {
    return Object.assign({}, timestamp)
  }

  // 0 pad numbers
  function padNumber (x, length) {
    var padded = String(x);
    while (padded.length < length) {
      padded = '0' + padded;
    }

    return padded
  }

  // get date in YYYY-MM-DD format
  function getDate (timestamp) {
    var str = (padNumber(timestamp.year, 4)) + "-" + (padNumber(timestamp.month, 2));

    if (timestamp.hasDay) { str += "-" + (padNumber(timestamp.day, 2)); }

    return str
  }

  // get time in HH:mm format
  function getTime (timestamp) {
    if (!timestamp.hasTime) {
      return ''
    }

    return ((padNumber(timestamp.hour, 2)) + ":" + (padNumber(timestamp.minute, 2)))
  }

  // get date/time in "YYYY-MM-DD HH:mm" format
  function getDateTime (timestamp) {
    return getDate(timestamp) + ' ' + (timestamp.hasTime ? getTime(timestamp) : '00:00')
  }

  // returns timestamp of next day from passed timestamp
  function nextDay (timestamp) {
    ++timestamp.day;
    timestamp.weekday = (timestamp.weekday + 1) % DAYS_IN_WEEK;
    if (timestamp.day > DAYS_IN_MONTH_MIN && timestamp.day > daysInMonth(timestamp.year, timestamp.month)) {
      timestamp.day = DAY_MIN;
      ++timestamp.month;
      if (timestamp.month > MONTH_MAX) {
        timestamp.month = MONTH_MIN;
        ++timestamp.year;
      }
    }

    return timestamp
  }

  // returns timestamp of previous day from passed timestamp
  function prevDay (timestamp) {
    timestamp.day--;
    timestamp.weekday = (timestamp.weekday + 6) % DAYS_IN_WEEK;
    if (timestamp.day < DAY_MIN) {
      timestamp.month--;
      if (timestamp.month < MONTH_MIN) {
        timestamp.year--;
        timestamp.month = MONTH_MAX;
      }
      timestamp.day = daysInMonth(timestamp.year, timestamp.month);
    }

    return timestamp
  }

  // Giving a name that is easier to userstand
  function moveRelativeDays (timestamp, mover, days, allowedWeekdays) {
    if ( mover === void 0 ) mover = nextDay;
    if ( days === void 0 ) days = 1;
    if ( allowedWeekdays === void 0 ) allowedWeekdays = [0, 1, 2, 3, 4, 5, 6];

    return relativeDays(timestamp, mover, days, allowedWeekdays)
  }

  // Keeping this one so nothing breaks
  function relativeDays (timestamp, mover, days, allowedWeekdays) {
    if ( mover === void 0 ) mover = nextDay;
    if ( days === void 0 ) days = 1;
    if ( allowedWeekdays === void 0 ) allowedWeekdays = [0, 1, 2, 3, 4, 5, 6];

    while (--days >= 0) {
      timestamp = mover(timestamp);
      if (allowedWeekdays.length < 7 && !allowedWeekdays.includes(timestamp.weekday)) {
        ++days;
      }
    }

    return timestamp
  }

  function findWeekday (timestamp, weekday, mover, maxDays) {
    if ( mover === void 0 ) mover = nextDay;
    if ( maxDays === void 0 ) maxDays = 6;

    while (timestamp.weekday !== weekday && --maxDays >= 0) { timestamp = mover(timestamp); }
    return timestamp
  }

  function getWeekdaySkips (weekdays) {
    var skips = [1, 1, 1, 1, 1, 1, 1];
    var filled = [0, 0, 0, 0, 0, 0, 0];
    for (var i = 0; i < weekdays.length; ++i) {
      filled[weekdays[i]] = 1;
    }
    for (var k = 0; k < DAYS_IN_WEEK; ++k) {
      var skip = 1;
      for (var j = 1; j < DAYS_IN_WEEK; ++j) {
        var next = (k + j) % DAYS_IN_WEEK;
        if (filled[next]) {
          break
        }
        ++skip;
      }
      skips[k] = filled[k] * skip;
    }

    return skips
  }

  // create a list of days based on passed criteria
  function createDayList (start, end, now, weekdaySkips, disabledBefore, disabledAfter, disabledWeekdays, disabledDays, max, min) {
    if ( disabledWeekdays === void 0 ) disabledWeekdays = [];
    if ( disabledDays === void 0 ) disabledDays = [];
    if ( max === void 0 ) max = 42;
    if ( min === void 0 ) min = 0;

    var stop = getDayIdentifier(end);
    var days = [];
    var current = copyTimestamp(start);
    var currentIdentifier = 0;
    var stopped = currentIdentifier === stop;

    if (stop < getDayIdentifier(start)) {
      return days
    }

    while ((!stopped || days.length < min) && days.length < max) {
      currentIdentifier = getDayIdentifier(current);
      stopped = stopped || (currentIdentifier > stop && days.length >= min);
      if (stopped) {
        break
      }
      if (weekdaySkips[current.weekday] === 0) {
        current = nextDay(current);
        continue
      }
      var day = copyTimestamp(current);
      updateFormatted(day);
      updateRelative(day, now);
      updateDisabled(day, disabledBefore, disabledAfter, disabledWeekdays, disabledDays);
      days.push(day);
      current = relativeDays(current, nextDay);
    }

    return days
  }

  // create a list of intervals based on passed criteria
  function createIntervalList (timestamp, first, minutes, count, now) {
    var intervals = [];

    for (var i = 0; i < count; ++i) {
      var mins = (first + i) * minutes;
      var int = copyTimestamp(timestamp);
      intervals.push(updateMinutes(int, mins, now));
    }

    return intervals
  }

  // create an Intl.DateTimeFormat formatter
  function createNativeLocaleFormatter (locale, getOptions) {
    var emptyFormatter = function (_t, _s) { return ''; };

    /* istanbul ignore next */
    if (typeof Intl === 'undefined' || typeof Intl.DateTimeFormat === 'undefined') {
      return emptyFormatter
    }

    return function (timestamp, short) {
      try {
        var intlFormatter = new Intl.DateTimeFormat(locale || undefined, getOptions(timestamp, short));
        return intlFormatter.format(makeDateTime(timestamp))
      }
      catch (e) /* istanbul ignore next */ {
        /* eslint-disable-next-line */
        console.error(("Intl.DateTimeFormat: " + (e.message) + " -> " + (getDateTime(timestamp))));
        return ''
      }
    }
  }

  // return a JavaScript Date from passed timestamp
  function makeDate (timestamp) {
    return new Date(Date.UTC(timestamp.year, timestamp.month - 1, timestamp.day, 0, 0))
  }

  // return a JavaScript Date (with time) from passed timestamp
  function makeDateTime (timestamp) {
    return new Date(Date.UTC(timestamp.year, timestamp.month - 1, timestamp.day, timestamp.hour, timestamp.minute))
  }

  // validate a number IS a number
  function validateNumber (input) {
    return isFinite(parseInt(input, 10))
  }

  // check if passed timestamp is between two other timestamps
  function isBetweenDates (timestamp, startTimestamp, endTimestamp, useTime /* = false */) {
    var cd = getDayIdentifier(timestamp) + (useTime === true ? getTimeIdentifier(timestamp) : 0);
    var sd = getDayIdentifier(startTimestamp) + (useTime === true ? getTimeIdentifier(startTimestamp) : 0);
    var ed = getDayIdentifier(endTimestamp) + (useTime === true ? getTimeIdentifier(endTimestamp) : 0);

    return cd >= sd && cd <= ed
  }

  // check if two sets of start/end timestamps overlap each other
  function isOverlappingDates (startTimestamp, endTimestamp, firstTimestamp, lastTimestamp) {
    var start = getDayIdentifier(startTimestamp);
    var end = getDayIdentifier(endTimestamp);
    var first = getDayIdentifier(firstTimestamp);
    var last = getDayIdentifier(lastTimestamp);
    return (
      (start >= first && start <= last) || // overlap left
      (end >= first && end <= last) || // overlap right
      (first >= start && end >= last) // surrounding
    )
  }

  // add years, months, days, hours or minutes to a timestamp
  function addToDate (timestamp, options) {
    var ts = copyTimestamp(timestamp);
    var minType;
    __forEachObject(options, function (value, key) {
      if (ts[key] !== undefined) {
        ts[key] += parseInt(value, 10);
        var indexType = NORMALIZE_TYPES.indexOf(key);
        if (indexType !== -1) {
          if (minType === undefined) {
            minType = indexType;
          }
          else {
            /* istanbul ignore next */
            minType = Math.min(indexType, minType);
          }
        }
      }
    });

    // normalize timestamp
    if (minType !== undefined) {
      __normalize(ts, NORMALIZE_TYPES[minType]);
    }
    updateFormatted(ts);
    return ts
  }

  var NORMALIZE_TYPES = ['minute', 'hour', 'day', 'month'];

  // addToDate helper
  function __forEachObject (obj, cb) {
    Object.keys(obj).forEach(function (k) { return cb(obj[k], k); });
  }

  // normalize minutes
  function __normalizeMinute (ts) {
    if (ts.minute >= MINUTES_IN_HOUR || ts.minute < 0) {
      var hours = Math.floor(ts.minute / MINUTES_IN_HOUR);
      ts.minute -= hours * MINUTES_IN_HOUR;
      ts.hour += hours;
      __normalizeHour(ts);
    }
    return ts
  }

  // normalize hours
  function __normalizeHour (ts) {
    if (ts.hour >= HOURS_IN_DAY || ts.hour < 0) {
      var days = Math.floor(ts.hour / HOURS_IN_DAY);
      ts.hour -= days * HOURS_IN_DAY;
      ts.day += days;
      __normalizeDay(ts);
    }
    return ts
  }

  // normalize days
  function __normalizeDay (ts) {
    __normalizeMonth(ts);
    var dim = daysInMonth(ts.year, ts.month);
    if (ts.day > dim) {
      ++ts.month;
      if (ts.month > MONTH_MAX) {
        __normalizeMonth(ts);
      }
      var days = ts.day - dim;
      dim = daysInMonth(ts.year, ts.month);
      do {
        if (days > dim) {
          ++ts.month;
          if (ts.month > MONTH_MAX) {
            __normalizeMonth(ts);
          }
          days -= dim;
          dim = daysInMonth(ts.year, ts.month);
        }
      } while (days > dim)
      ts.day = days;
    }
    else if (ts.day <= 0) {
      var days$1 = -1 * ts.day;
      --ts.month;
      if (ts.month <= 0) {
        __normalizeMonth(ts);
      }
      dim = daysInMonth(ts.year, ts.month);
      do {
        if (days$1 > dim) /* istanbul ignore next */ {
          days$1 -= dim;
          --ts.month;
          if (ts.month <= 0) {
            __normalizeMonth(ts);
          }
          dim = daysInMonth(ts.year, ts.month);
        }
      } while (days$1 > dim)
      ts.day = dim - days$1;
    }
    return ts
  }

  // normalize months
  function __normalizeMonth (ts) {
    if (ts.month > MONTH_MAX) {
      var years = Math.floor(ts.month / MONTH_MAX);
      ts.month = ts.month % MONTH_MAX;
      ts.year += years;
    }
    else if (ts.month < MONTH_MIN) {
      ts.month += MONTH_MAX;
      --ts.year;
    }
    return ts
  }

  // normalize all
  function __normalize (ts, type) {
    switch (type) {
      case 'minute':
        return __normalizeMinute(ts)
      case 'hour':
        return __normalizeHour(ts)
      case 'day':
        return __normalizeDay(ts)
      case 'month':
        return __normalizeMonth(ts)
    }
  }

  // returns the number of days between two timestamps
  function daysBetween (ts1, ts2) {
    var diff = diffTimestamp(ts1, ts2, true);
    return Math.floor(diff / MILLISECONDS_IN_DAY)
  }

  // returns the number of weeks between two timestamps
  function weeksBetween (ts1, ts2) {
    var t1 = copyTimestamp(ts1);
    var t2 = copyTimestamp(ts2);
    t1 = findWeekday(t1, 0);
    t2 = findWeekday(t2, 6);
    return Math.ceil(daysBetween(t1, t2) / DAYS_IN_WEEK)
  }

  var Times = {
    name: 'Times',

    props: {
      now: {
        type: String,
        validator: function (v) { return v === '' || validateTimestamp(v); },
        default: ''
      }
    },

    data: function () { return ({
      times: {
        now: parseTimestamp('0000-00-00 00:00'),
        today: parseTimestamp('0000-00-00')
      }
    }); },

    computed: {
      parsedNow: function parsedNow () {
        return this.now ? parseTimestamp(this.now) : this.getNow()
      }
    },

    watch: {
      parsedNow: 'updateCurrent'
    },

    beforeMount: function beforeMount () {
      this.updateCurrent();
      this.setCurrent();
    },

    methods: {
      setCurrent: function setCurrent () {
        this.times.now.current = this.times.today.current = true;
        this.times.now.past = this.times.today.past = false;
        this.times.now.future = this.times.today.future = false;
      },

      updateCurrent: function updateCurrent () {
        var now = this.parsedNow || this.getNow();
        this.updateDay(now, this.times.now);
        this.updateTime(now, this.times.now);
        this.updateDay(now, this.times.today);
      },

      getNow: function getNow () {
        return parseDate(new Date())
      },

      updateDay: function updateDay (now, target) {
        if (now.date !== target.date) {
          target.year = now.year;
          target.month = now.month;
          target.day = now.day;
          target.weekday = now.weekday;
          target.date = now.date;
        }
      },

      updateTime: function updateTime (now, target) {
        if (now.time !== target.time) {
          target.hour = now.hour;
          target.minute = now.minute;
          target.time = now.time;
        }
      }
    }
  };

  var Mouse = {
    name: 'Mouse',

    methods: {

      getDefaultMouseEventWithSuffix: function getDefaultMouseEventWithSuffix (suffix) {
        var obj;

        return ( obj = {}, obj['click' + suffix] = { event: 'click' }, obj['contextmenu' + suffix] = { event: 'contextmenu', prevent: true, result: false }, obj['mousedown' + suffix] = { event: 'mousedown' }, obj['mousemove' + suffix] = { event: 'mousemove' }, obj['mouseup' + suffix] = { event: 'mouseup' }, obj['mouseenter' + suffix] = { event: 'mouseenter' }, obj['mouseleave' + suffix] = { event: 'mouseleave' }, obj['touchstart' + suffix] = { event: 'touchstart' }, obj['touchmove' + suffix] = { event: 'touchmove' }, obj['touchend' + suffix] = { event: 'touchend' }, obj )
      },

      getDefaultMouseEventHandlers2: function getDefaultMouseEventHandlers2 (suffix1, suffix2, getEvent) {
        var obj;

        return this.getMouseEventHandlers(( obj = {}, obj['click' + suffix1] = { event: 'click' }, obj['contextmenu' + suffix1] = { event: 'contextmenu', prevent: true, result: false }, obj['mousedown' + suffix1] = { event: 'mousedown' }, obj['mousemove' + suffix1] = { event: 'mousemove' }, obj['mouseup' + suffix1] = { event: 'mouseup' }, obj['mouseenter' + suffix1] = { event: 'mouseenter' }, obj['mouseleave' + suffix1] = { event: 'mouseleave' }, obj['touchstart' + suffix1] = { event: 'touchstart' }, obj['touchmove' + suffix1] = { event: 'touchmove' }, obj['touchend' + suffix1] = { event: 'touchend' }, obj['click' + suffix2] = { event: 'click' }, obj['contextmenu' + suffix2] = { event: 'contextmenu', prevent: true, result: false }, obj['mousedown' + suffix2] = { event: 'mousedown' }, obj['mousemove' + suffix2] = { event: 'mousemove' }, obj['mouseup' + suffix2] = { event: 'mouseup' }, obj['mouseenter' + suffix2] = { event: 'mouseenter' }, obj['mouseleave' + suffix2] = { event: 'mouseleave' }, obj['touchstart' + suffix2] = { event: 'touchstart' }, obj['touchmove' + suffix2] = { event: 'touchmove' }, obj['touchend' + suffix2] = { event: 'touchend' }, obj ), getEvent)
      },

      getDefaultMouseEventHandlers: function getDefaultMouseEventHandlers (suffix, getEvent) {
        var obj;

        return this.getMouseEventHandlers(( obj = {}, obj['click' + suffix] = { event: 'click' }, obj['contextmenu' + suffix] = { event: 'contextmenu', prevent: true, result: false }, obj['mousedown' + suffix] = { event: 'mousedown' }, obj['mousemove' + suffix] = { event: 'mousemove' }, obj['mouseup' + suffix] = { event: 'mouseup' }, obj['mouseenter' + suffix] = { event: 'mouseenter' }, obj['mouseleave' + suffix] = { event: 'mouseleave' }, obj['touchstart' + suffix] = { event: 'touchstart' }, obj['touchmove' + suffix] = { event: 'touchmove' }, obj['touchend' + suffix] = { event: 'touchend' }, obj ), getEvent)
      },
      getMouseEventHandlers: function getMouseEventHandlers (events, getEvent) {
        var this$1$1 = this;

        var on = {};

        var loop = function ( eventName ) {
          var eventOptions = events[eventName];

          if (!this$1$1.$listeners[eventName]) { return }

          // https://vuejs.org/v2/guide/render-function.html#Event-amp-Key-Modifiers
          var prefix = eventOptions.passive ? '&' : ((eventOptions.once ? '~' : '') + (eventOptions.capture ? '!' : ''));
          var key = prefix + eventOptions.event;

          var handler = function (event) {
            var mouseEvent = event;
            if (eventOptions.button === undefined || (mouseEvent.buttons > 0 && mouseEvent.button === eventOptions.button)) {
              if (eventOptions.prevent) {
                mouseEvent.preventDefault();
              }
              if (eventOptions.stop) {
                mouseEvent.stopPropagation();
              }
              this$1$1.$emit(eventName, getEvent(mouseEvent, eventName));
            }

            return eventOptions.result
          };

          if (key in on) {
            if (Array.isArray(on[key])) {
              (on[key]).push(handler);
            }
            else {
              on[key] = [on[key], handler];
            }
          }
          else {
            on[key] = handler;
          }
        };

        for (var eventName in events) loop( eventName );

        return on
      }
    }
  };

  function validateView (view) {
    return [
      'month', 'week', 'day',
      '2day', '3day', '4day', '5day', '6day',
      'month-scheduler', 'week-scheduler', 'custom-scheduler',
      'scheduler', 'day-scheduler', '2day-scheduler', '3day-scheduler',
      '4day-scheduler', '5day-scheduler', '6day-scheduler',
      'resource', 'day-resource',
      'month-agenda', 'week-agenda', 'custom-agenda',
      'agenda', 'day-agenda', '2day-agenda', '3day-agenda',
      '4day-agenda', '5day-agenda', '6day-agenda',
      'month-interval', 'custom-interval'].includes(view)
  }

  /* public properties */
  var props = {
    base: {
      value: { // v-model
        type: String,
        validator: function (v) { return v === '' || validateTimestamp(v); }
      },
      weekdays: {
        type: Array,
        default: function () { return [0, 1, 2, 3, 4, 5, 6]; }
      },
      noActiveDate: Boolean,
      disabledDays: Array,
      disabledBefore: String,
      disabledAfter: String,
      disabledWeekdays: {
        type: Array,
        default: function () { return []; }
      },
      hideHeader: Boolean,
      noScroll: Boolean,
      shortWeekdayLabel: Boolean,
      noDefaultHeaderText: Boolean,
      noDefaultHeaderBtn: Boolean,
      locale: {
        type: String,
        default: 'en-us'
      },
      animated: Boolean,
      transitionPrev: {
        type: String,
        default: 'slide-right'
      },
      transitionNext: {
        type: String,
        default: 'slide-left'
      },
      dragOverFunc: {
        type: Function
        // event, timestamp
      },
      dropFunc: {
        type: Function
        // event, timestamp
      },
      selectedDates: Array,
      selectedStartEndDates: {
        type: Array,
        default: function () { return []; },
        validator: function (v) { return v.length <= 2; }
      }
    },
    intervals: {
      maxDays: {
        type: Number,
        default: 7
      },
      shortIntervalLabel: Boolean,
      intervalHeight: {
        type: [Number, String],
        default: 40,
        validator: validateNumber
      },
      intervalMinutes: {
        type: [Number, String],
        default: 60,
        validator: validateNumber
      },
      intervalStart: {
        type: [Number, String],
        default: 0,
        validator: validateNumber
      },
      intervalCount: {
        type: [Number, String],
        default: 24,
        validator: validateNumber
      },
      intervalStyle: {
        type: Function,
        default: null
      },
      showIntervalLabel: {
        type: Function,
        default: null
      },
      hour24Format: Boolean,
      columnHeaderBefore: Boolean,
      columnHeaderAfter: Boolean,
      columnCount: {
        type: [Number, String],
        default: 1,
        validator: validateNumber
      },
      columnIndexStart: {
        type: [Number, String],
        default: 0,
        validator: validateNumber
      },
      timeClicksClamped: Boolean
    },
    weeks: {
      dayHeight: {
        type: [Number, String],
        default: 0,
        validator: validateNumber
      },
      dayStyle: {
        type: Function,
        default: null
      },
      dayClass: {
        type: Function,
        default: null
      },
      dayPadding: String,
      minWeeks: {
        type: [Number, String],
        validator: validateNumber,
        default: 1
      },
      shortMonthLabel: Boolean,
      showWorkWeeks: Boolean,
      showMonthLabel: {
        type: Boolean,
        default: true
      },
      showDayOfYearLabel: Boolean,
      enableOutsideDays: Boolean,
      hideOutsideDays: Boolean,
      hover: Boolean,
      miniMode: {
        type: [Boolean, String],
        validator: function (v) { return v === undefined || v === true || v === false || v === 'auto'; }
      },
      breakpoint: {
        type: String,
        default: 'md',
        validator: function (v) { return ['xs', 'sm', 'md', 'lg', 'xl'].includes(v); }
      },
      monthLabelSize: {
        type: String,
        default: 'sm',
        validator: function (v) { return ['xs', 'sm', 'md', 'lg', 'xl'].includes(v); }
      }
    },
    scheduler: {
      resources: Array,
      resourceKey: {
        type: String,
        default: 'label'
      },
      maxDays: {
        type: Number,
        default: 7
      },
      resourceHeight: {
        type: [Number, String],
        default: 70,
        validator: validateNumber
      },
      resourceWidth: {
        type: [Number, String],
        validator: function (v) { return v === undefined || validateNumber(v); }
      },
      resourceStyle: {
        type: Function,
        default: null
      },
      cellWidth: String,
      columnHeaderBefore: Boolean,
      columnHeaderAfter: Boolean,
      columnCount: {
        type: [Number, String],
        default: 1,
        validator: validateNumber
      },
      columnIndexStart: {
        type: [Number, String],
        default: 0,
        validator: validateNumber
      }
    },
    resource: {
      resources: Array,
      resourceKey: {
        type: String,
        default: 'label'
      },
      resourceHeight: {
        type: [Number, String],
        default: 70,
        validator: validateNumber
      },
      resourceWidth: {
        type: [Number, String],
        default: 100,
        validator: function (v) { return v === undefined || validateNumber(v); }
      },
      resourceStyle: {
        type: Function,
        default: null
      },
      intervalWidth: {
        type: [Number, String],
        default: 100,
        validator: validateNumber
      },
      intervalHeight: {
        type: [Number, String],
        default: 20,
        validator: validateNumber
      },
      sticky: Boolean
    },
    agenda: {
      leftColumnOptions: Array,
      rightColumnOptions: Array,
      columnOptionsId: String,
      columnOptionsLabel: String
    },
    calendar: {
      view: {
        type: String,
        default: 'month',
        validator: validateView
      },
      bordered: Boolean,
      dark: Boolean
    }
  };

  // Mixins

  var CalendarBase = {
    name: 'CalendarBase',

    mixins: [
      Mouse,
      Times
    ],

    props: Object.assign({}, props.base,
      {start: {
        type: String,
        validator: validateTimestamp,
        default: function () { return parseDate(new Date()).date; }
      },
      end: {
        type: String,
        validator: validateTimestamp,
        default: '0000-00-00'
      }}),

    data: function data () {
      return {
        keyValue: 0
      }
    },

    computed: {
      weekdaySkips: function weekdaySkips () {
        return getWeekdaySkips(this.weekdays)
      },

      parsedStart: function parsedStart () {
        return parseTimestamp(this.start)
      },

      parsedEnd: function parsedEnd () {
        return parseTimestamp(this.end)
      },

      days: function days () {
        return createDayList(
          this.parsedStart,
          this.parsedEnd,
          this.times.today,
          this.weekdaySkips,
          this.disabledBefore,
          this.disabledAfter,
          this.disabledWeekdays,
          this.disabledDays
        )
      },

      dayFormatter: function dayFormatter () {
        var options = { timeZone: 'UTC', day: 'numeric' };

        return createNativeLocaleFormatter(
          this.locale,
          function (_tms, _short) { return options; }
        )
      },

      weekdayFormatter: function weekdayFormatter () {
        var longOptions = { timeZone: 'UTC', weekday: 'long' };
        var shortOptions = { timeZone: 'UTC', weekday: 'short' };

        return createNativeLocaleFormatter(
          this.locale,
          function (_tms, short) { return short ? shortOptions : longOptions; }
        )
      }
    },

    methods: {
      arrayHasDate: function arrayHasDate (arr, timestamp) {
        return arr && arr.length > 0 && arr.includes(timestamp.date)
      },

      arrayHasDateTime: function arrayHasDateTime (arr, timestamp) {
        return arr && arr.length > 0 && arr.includes(getDateTime(timestamp))
      },

      checkDays: function checkDays (arr, timestamp) {
        var days = {
          firstDay: false,
          betweenDays: false,
          lastDay: false
        };

        // array must have two dates ('YYYY-MM-DD') in it
        if (arr && arr.length === 2) {
          var current = getDayIdentifier(timestamp);
          var first = getDayIdentifier(parsed(arr[0]));
          var last = getDayIdentifier(parsed(arr[1]));
          days.firstDay = first === current;
          days.lastDay = last === current;
          days.betweenDays = first < current && last > current;
        }
        return days
      },

      checkIntervals: function checkIntervals (arr, timestamp) {
        var days = {
          firstDay: false,
          betweenDays: false,
          lastDay: false
        };

        // array must have two dates ('YYYY-MM-DD HH:MM') in it
        if (arr && arr.length === 2) {
          var current = getDayTimeIdentifier(timestamp);
          var first = getDayTimeIdentifier(parsed(arr[0]));
          var last = getDayTimeIdentifier(parsed(arr[1]));
          days.firstDay = first === current;
          days.lastDay = last === current;
          days.betweenDays = first < current && last > current;
        }
        return days
      },

      getRelativeClasses: function getRelativeClasses (timestamp, outside, selectedDays, startEndDays, hover) {
        if ( outside === void 0 ) outside = false;
        if ( selectedDays === void 0 ) selectedDays = [];
        if ( startEndDays === void 0 ) startEndDays = [];
        if ( hover === void 0 ) hover = false;

        var isSelected = this.arrayHasDate(selectedDays, timestamp);
        var ref = this.checkDays(startEndDays, timestamp);
        var firstDay = ref.firstDay;
        var lastDay = ref.lastDay;
        var betweenDays = ref.betweenDays;

        return {
          'q-past-day': firstDay !== true && betweenDays !== true && lastDay !== true && isSelected !== true && outside !== true && timestamp.past,
          'q-future-day': firstDay !== true && betweenDays !== true && lastDay !== true && isSelected !== true && outside !== true && timestamp.future,
          'q-outside': outside, // outside the current month
          'q-current-day': timestamp.current,
          'q-selected': isSelected,
          'q-range-first': firstDay === true,
          'q-range': betweenDays === true,
          'q-range-last': lastDay === true,
          'q-range-hover': hover === true && (firstDay === true || lastDay === true || betweenDays === true),
          'q-disabled-day disabled': timestamp.disabled === true
        }
      },

      getIntervalClasses: function getIntervalClasses (interval, selectedDays, startEndDays) {
        if ( selectedDays === void 0 ) selectedDays = [];
        if ( startEndDays === void 0 ) startEndDays = [];

        var isSelected = this.arrayHasDateTime(selectedDays, interval);
        var ref = this.checkIntervals(startEndDays, interval);
        var firstDay = ref.firstDay;
        var lastDay = ref.lastDay;
        var betweenDays = ref.betweenDays;

        return {
          'q-selected': isSelected,
          'q-range-first': firstDay === true,
          'q-range': betweenDays === true,
          'q-range-last': lastDay === true,
          'q-disabled-interval disabled': interval.disabled === true
        }
      },

      getStartOfWeek: function getStartOfWeek$1 (timestamp) {
        return getStartOfWeek(timestamp, this.weekdays, this.times.today)
      },

      getEndOfWeek: function getEndOfWeek$1 (timestamp) {
        return getEndOfWeek(timestamp, this.weekdays, this.times.today)
      },

      dayStyleDefault: function dayStyleDefault (_timestamp) {
        return undefined
      }
    }
  };

  function convertToUnit (input, unit) {
    if ( unit === void 0 ) unit = 'px';

    if (input == null || input === '') {
      return undefined
    }
    else if (isNaN(input)) {
      return String(input)
    }
    else if (input === 'auto') {
      return input
    }
    else {
      return ("" + (Number(input)) + unit)
    }
  }

  function indexOf (array, cb) {
    for (var i = 0; i < array.length; i++) {
      if (cb(array[i], i) === true) {
        return i
      }
    }
    return -1
  }

  // Quasar

  var QCalendarWeekly = {
    name: 'QCalendarWeekly',

    mixins: [
      CalendarBase
    ],

    props: Object.assign({}, props.weeks,
      {direction: {
        type: String,
        default: 'next'
      }}),

    mounted: function mounted () {
      this.__adjustForWeekEvents();
    },

    updated: function updated () {
      this.__adjustForWeekEvents();
    },

    computed: {
      staticClass: function staticClass () {
        return 'q-calendar-weekly'
      },

      parsedMinWeeks: function parsedMinWeeks () {
        return parseInt(this.minWeeks, 10)
      },

      days: function days () {
        var minDays = this.parsedMinWeeks * this.weekdays.length;
        var start = this.getStartOfWeek(this.parsedStart);
        var end = this.getEndOfWeek(this.parsedEnd);

        return createDayList(
          start,
          end,
          this.times.today,
          this.weekdaySkips,
          this.disabledBefore,
          this.disabledAfter,
          this.disabledWeekdays,
          this.disabledDays,
          Number.MAX_SAFE_INTEGER,
          minDays
        )
      },

      todayWeek: function todayWeek () {
        var day = this.days[0];
        var start = this.getStartOfWeek(day);
        var end = this.getEndOfWeek(day);

        return createDayList(
          start,
          end,
          day,
          this.weekdaySkips,
          this.disabledBefore,
          this.disabledAfter,
          this.disabledWeekdays,
          this.disabledDays,
          this.weekdays.length,
          this.weekdays.length
        )
      },

      monthFormatter: function monthFormatter () {
        var longOptions = { timeZone: 'UTC', month: 'long' };
        var shortOptions = { timeZone: 'UTC', month: 'short' };

        return createNativeLocaleFormatter(
          this.locale,
          function (_tms, short) { return short ? shortOptions : longOptions; }
        )
      },

      styles: function styles () {
        var style = {};
        if (this.dayPadding !== undefined) {
          style.padding = this.dayPadding;
        }
        style.minWidth = this.cellWidth + '%';
        style.maxWidth = style.minWidth;
        return style
      },

      cellWidth: function cellWidth () {
        return 100 / this.weekdays.length
      },

      isMiniMode: function isMiniMode () {
        return this.miniMode === true ||
          (this.miniMode === 'auto' && this.breakpoint !== void 0 && this.$q.screen.lt[this.breakpoint])
      }
    },

    watch: {
      isMinimode: function isMinimode (val) {
        if (this.breakpoint === 'auto') {
          this.$emit('mini-mode', val);
        }
      }
    },

    methods: {
      isOutside: function isOutside (day) {
        var dayIdentifier = getDayIdentifier(day);

        return dayIdentifier < getDayIdentifier(this.parsedStart) ||
               dayIdentifier > getDayIdentifier(this.parsedEnd)
      },

      isCurrentWeek: function isCurrentWeek (week) {
        for (var i = 0; i < week.length; ++i) {
          if (week[i].current === true) {
            return { timestamp: week[i] }
          }
        }
        return { timestamp: false }
      },

      __adjustForWeekEvents: function __adjustForWeekEvents () {
        if (this.isMiniMode === true) { return }
        if (this.dayHeight !== 0) { return }
        var slotWeek = this.$scopedSlots.week;
        if (slotWeek === void 0) { return }

        var i = 0;
        for (; i < 7; ++i) {
          var weekEvent = this.$refs['weekEvent' + i];
          if (weekEvent === void 0) { return }
          var wrapper = this.$refs['week' + i];
          if (wrapper === void 0) { return }
          // this sucks to have to do it this way
          var styles = window.getComputedStyle(weekEvent);
          var margin = parseFloat(styles.marginTop, 10) + parseFloat(styles.marginBottom, 10);
          if (weekEvent.clientHeight + margin > wrapper.clientHeight) {
            wrapper.style.height = weekEvent.clientHeight + margin + 'px';
          }
        }
      },

      __renderContainer: function __renderContainer (h) {
        var component = h('div', {
          staticClass: 'q-calendar-weekly__container'
        }, [].concat( this.__renderWeeks(h) ));

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }

        return component
      },

      __renderHead: function __renderHead (h) {
        return h('div', {
          staticClass: 'q-calendar-weekly__head'
        }, [
          h('div', {
            staticClass: 'q-calendar-weekly__head--wrapper'
          }, [
            this.showWorkWeeks === true && this.__renderWorkWeekHead(h),
            this.__renderHeadDays(h)
          ])
        ])
      },

      __renderWorkWeekHead: function __renderWorkWeekHead (h) {
        var slot = this.$scopedSlots['workweek-header'];
        var scope = {
          start: this.parsedStart,
          end: this.parsedEnd,
          miniMode: this.isMiniMode
        };

        return h('div', {
          staticClass: 'q-calendar-weekly__head-workweek',
          on: this.getDefaultMouseEventHandlers(':workweek:header2', function (event) {
            return { scope: scope, event: event }
          })
        }, (slot ? slot({ scope: scope }) : '#'))
      },

      __renderHeadDays: function __renderHeadDays (h) {
        var this$1$1 = this;

        return h('div', {
          staticClass: 'q-calendar-weekly__head-weekdays',
          style: {
            minWidth: this.showWorkWeeks
              ? (this.isMiniMode === true
                  ? 'calc(100% - var(--calendar-mini-work-week-width))'
                  : 'calc(100% - var(--calendar-work-week-width))')
              : '100%'
          }
        }, [].concat( this.todayWeek.map(function (day, index) { return this$1$1.__renderHeadDay(h, day, index); }) ))
      },

      __renderHeadDay: function __renderHeadDay (h, day, index) {
        var width = this.cellWidth + '%';
        var headDaySlot = this.$scopedSlots['head-day'];
        var disabled = (this.disabledWeekdays ? this.disabledWeekdays.includes(day.weekday) : false);
        var days = this.days.filter(function (day2) { return day2.weekday === day.weekday; });
        var scope = { timestamp: day, days: days, index: index, miniMode: this.isMiniMode };

        return h('div', {
          key: day.date,
          staticClass: 'q-calendar-weekly__head-weekday' + (disabled === true ? ' q-disabled-day disabled' : ''),
          style: {
            minWidth: width,
            maxWidth: width
          },
          on: this.getDefaultMouseEventHandlers(':day:header2', function (event) {
            return { scope: scope, event: event }
          })
        }, [
          headDaySlot === undefined && this.__renderHeadDayLabel(h, day, this.shortWeekdayLabel || this.isMiniMode),
          headDaySlot !== undefined && headDaySlot(scope)
        ])
      },

      __renderHeadDayLabel: function __renderHeadDayLabel (h, day, shortWeekdayLabel) {
        var weekdayLabel = this.weekdayFormatter(day, shortWeekdayLabel);
        return h('span', {
          staticClass: 'ellipsis'
        }, this.isMiniMode === true && this.shortWeekdayLabel === true ? weekdayLabel.charAt(0) : weekdayLabel)
      },

      __renderWeeks: function __renderWeeks (h) {
        var days = this.days;
        var weekDays = this.weekdays.length;
        var weeks = [];
        for (var i = 0; i < days.length; i += weekDays) {
          weeks.push(this.__renderWeek(h, days.slice(i, i + weekDays), i / weekDays));
        }

        return weeks
      },

      __renderWeek: function __renderWeek (h, week, weekNum) {
        var this$1$1 = this;

        var slotWeek = this.$scopedSlots.week;
        var weekdays = this.weekdays;
        var slotData = { week: week, weekdays: weekdays, miniMode: this.isMiniMode };
        var style = {};

        // this applies height properly, even if workweeks are displaying
        style.height = this.dayHeight > 0 ? convertToUnit(this.dayHeight) : 'auto';

        return h('div', {
          key: week[0].date,
          ref: 'week' + weekNum,
          staticClass: 'q-calendar-weekly__week--wrapper',
          style: style
        }, [
          this.showWorkWeeks === true && this.__renderWorkWeekGutter(h, week),
          h('div', {
            staticClass: 'q-calendar-weekly__week',
            style: {
              width: this.showWorkWeeks
                ? (this.isMiniMode === true
                    ? 'calc(100% - var(--calendar-mini-work-week-width))'
                    : 'calc(100% - var(--calendar-work-week-width))')
                : '100%'
            }
          }, [
            h('div', {
              staticClass: 'q-calendar-weekly__week-days'
            }, week.map(function (day) { return this$1$1.__renderDay(h, day); })),
            this.isMiniMode !== true && slotWeek !== undefined
              ? h('div', {
                  ref: 'weekEvent' + weekNum,
                  staticClass: 'q-calendar-weekly__week-events'
                }, slotWeek(slotData))
              : ''
          ])
        ])
      },

      __renderWorkWeekGutter: function __renderWorkWeekGutter (h, week) {
        var slot = this.$scopedSlots.workweek;
        // adjust day to account for Sunday/Monday start of week calendars
        var day = week.length > 2 ? week[2] : week[0];
        var ref = this.isCurrentWeek(week);
        var timestamp = ref.timestamp;
        var workweekLabel = Number(day.workweek).toLocaleString(this.locale);
        var scope = { workweekLabel: workweekLabel, week: week, miniMode: this.isMiniMode };

        return h('div', {
          key: day.workweek,
          staticClass: 'q-calendar-weekly__workweek',
          class: this.getRelativeClasses(timestamp !== false ? timestamp : day, false),
          on: this.getDefaultMouseEventHandlers(':workweek2', function (event) {
            return { scope: scope, event: event }
          })
        }, slot ? slot(scope) : workweekLabel)
      },

      __renderDay: function __renderDay (h, day) {
        var this$1$1 = this;

        var styler = this.dayStyle || this.dayStyleDefault;
        var outside = this.isOutside(day);
        var activeDate = this.noActiveDate !== true && this.value === day.date;
        var slot = this.$scopedSlots.day;
        var scope = { outside: outside, timestamp: day, miniMode: this.isMiniMode, activeDate: activeDate };
        var hasMonth = (outside === false && this.days.find(function (d) { return d.month === day.month; }).day === day.day && this.showMonthLabel === true);

        var dragOver;
        var style = Object.assign(Object.assign({}, this.styles), styler(day));
        var dayClass = typeof this.dayClass === 'function' ? this.dayClass(day) : null;

        return h('div', {
          key: day.date,
          staticClass: 'q-calendar-weekly__day' + (this.isMiniMode !== true ? ' column' : ''),
          class: Object.assign({}, dayClass,
            this.getRelativeClasses(day, outside, this.selectedDates, this.selectedStartEndDates, this.hover),
            {'q-active-date': activeDate === true,
            disabled: this.enableOutsideDays !== true && outside === true,
            'q-calendar-weekly__day--droppable': dragOver}),
          style: style,
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, day, 'day');
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, day, 'day');
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':day2', function (event) {
            return { scope: scope, event: event }
          })
        }, [
          this.__renderDayLabel(h, day),
          this.isMiniMode !== true && this.showDayOfYearLabel && !hasMonth ? this.__renderDayOfYearLabel(h, day) : '',
          this.isMiniMode !== true && hasMonth ? this.__renderDayMonth(h, day) : '',
          h('div', {
            staticClass: 'q-calendar-weekly__day--content full-width' + (this.isMiniMode === true ? ' row justify-center items-center' : '')
          }, slot ? slot(scope) : '')
        ])
      },

      __renderDayLabel: function __renderDayLabel (h, day) {
        var outside = this.isOutside(day);

        // return if outside days are hidden
        if (outside === true && this.hideOutsideDays === true) {
          return
        }

        var dayLabel = this.dayFormatter(day, false);
        var dayLabelSlot = this.$scopedSlots['day-label'];
        var dayBtnSlot = this.$scopedSlots['day-btn'];

        var selectedDate = (
          // this.isMiniMode &&
          this.selectedDates &&
          this.selectedDates.length > 0 &&
          this.selectedDates.includes(day.date)
        );

        var activeDate = this.noActiveDate !== true && this.value === day.date;
        var slotData = { dayLabel: dayLabel, timestamp: day, outside: outside, activeDate: activeDate, selectedDate: selectedDate, miniMode: this.isMiniMode };

        return dayBtnSlot
          ? dayBtnSlot(slotData)
          : h(quasar.QBtn, {
            staticClass: 'q-calendar-weekly__day-label',
            props: {
              size: this.isMiniMode ? 'sm' : this.monthLabelSize,
              unelevated: true,
              round: true,
              dense: true,
              noCaps: true,
              outline: day.current === true,
              disable: day.disabled === true || (this.enableOutsideDays !== true && outside === true)
            },
            style: {
              lineHeight: this.isMiniMode ? 'unset' : '1.715em'
            },
            on: this.getDefaultMouseEventHandlers(':date2', function (event) {
              return { scope: { timestamp: day }, event: event }
            })
          }, [
            dayLabelSlot ? dayLabelSlot(slotData) : dayLabel
          ])
      },

      __renderDayOfYearLabel: function __renderDayOfYearLabel (h, day) {
        var outside = this.isOutside(day);

        // return if outside days are hidden
        if (outside === true && this.hideOutsideDays === true) {
          return
        }

        var slot = this.$scopedSlots['day-of-year'];
        var slotData = { timestamp: day };

        return h('div', {
          staticClass: 'q-calendar-weekly__day-month--day-of-year'
        }, slot ? slot(slotData) : day.doy)
      },

      __renderDayMonth: function __renderDayMonth (h, day) {
        var outside = this.isOutside(day);

        // return if outside days are hidden
        if (outside === true && this.hideOutsideDays === true) {
          return
        }

        var slot = this.$scopedSlots['month-label'];
        var monthLabel = this.monthFormatter(day, this.shortMonthLabel);
        var slotData = { monthLabel: monthLabel, timestamp: day, miniMode: this.isMiniMode };

        return h('div', {
          staticClass: 'q-calendar-weekly__day-month ellipsis'
        }, [
          slot ? slot(slotData) : this.isMiniMode !== true ? monthLabel : ''
        ])
      }
    },

    render: function render (h) {
      return h('div', {
        class: this.staticClass,
        on: {
          dragstart: function (e) {
            e.preventDefault();
          }
        }
      }, [
        !this.hideHeader && this.__renderHead(h),
        this.__renderContainer(h)
      ])
    }
  };

  // Mixins

  /* @vue/component */
  var QCalendarMonthly = {
    name: 'QCalendarMonthly',

    mixins: [
      QCalendarWeekly
    ],

    computed: {
      parsedStart: function parsedStart () {
        return getStartOfMonth(parseTimestamp(this.start))
      },

      parsedEnd: function parsedEnd () {
        return getEndOfMonth(parseTimestamp(this.end))
      }
    }
  };

  var Resize = {
    name: 'resize',

    inserted: function inserted (el, binding) {
      var callback = binding.value;
      var options = binding.options || { passive: true };

      window.addEventListener('resize', callback, options);
      el._onResize = {
        callback: callback,
        options: options
      };

      if (!binding.modifiers || !binding.modifiers.quiet) {
        callback();
      }
    },

    unbind: function unbind (el) {
      if (!el._onResize) { return }

      var ref = el._onResize;
      var callback = ref.callback;
      var options = ref.options;
      window.removeEventListener('resize', callback, options);
      delete el._onResize;
    }
  };

  // Mixins

  var CalendarIntervals = {
    name: 'CalendarWithIntervals',

    mixins: [
      CalendarBase
    ],

    props: Object.assign({}, props.intervals),

    computed: {
      parsedIntervalStart: function parsedIntervalStart () {
        return parseInt(this.intervalStart, 10)
      },

      parsedIntervalMinutes: function parsedIntervalMinutes () {
        return parseInt(this.intervalMinutes, 10)
      },

      parsedIntervalCount: function parsedIntervalCount () {
        return parseInt(this.intervalCount, 10)
      },

      parsedIntervalHeight: function parsedIntervalHeight () {
        return parseFloat(this.intervalHeight)
      },

      parsedIntervalWidth: function parsedIntervalWidth () {
        return parseFloat(this.intervalWidth)
      },

      startMinute: function startMinute () {
        return this.parsedIntervalStart * this.parsedIntervalMinutes
      },

      bodyHeight: function bodyHeight () {
        return this.parsedIntervalCount * this.parsedIntervalHeight
      },

      bodyWidth: function bodyWidth () {
        return this.parsedIntervalCount * this.parsedIntervalWidth
      },

      days: function days () {
        return createDayList(
          this.parsedStart,
          this.parsedEnd,
          this.times.today,
          this.weekdaySkips,
          this.disabledBefore,
          this.disabledAfter,
          this.disabledWeekdays,
          this.disabledDays,
          this.maxDays
        )
      },

      intervals: function intervals () {
        var days = this.days;
        var first = this.parsedIntervalStart;
        var minutes = this.parsedIntervalMinutes;
        var count = this.parsedIntervalCount;
        var now = this.times.now;

        if (days.length === 0) {
          /* eslint-disable no-console */
          console.error('QCalendar: days array has no content');
        }
        return days.map(function (d) { return createIntervalList(d, first, minutes, count, now); })
      },

      intervalFormatter: function intervalFormatter () {
        var longOptions = { timeZone: 'UTC', hour12: !this.hour24Format, hour: '2-digit', minute: '2-digit' };
        var shortOptions = { timeZone: 'UTC', hour12: !this.hour24Format, hour: 'numeric', minute: '2-digit' };
        var shortHourOptions = { timeZone: 'UTC', hour12: !this.hour24Format, hour: 'numeric' };

        return createNativeLocaleFormatter(
          this.locale,
          function (tms, short) { return short ? (tms.minute === 0 ? shortHourOptions : shortOptions) : longOptions; }
        )
      }
    },

    methods: {
      showIntervalLabelDefault: function showIntervalLabelDefault (interval) {
        var first = this.intervals[0][0];
        var isFirst = first.hour === interval.hour && first.minute === interval.minute;
        return !isFirst && interval.minute === 0
      },

      intervalStyleDefault: function intervalStyleDefault (_interval) {
        return undefined
      },

      getTimestampAtEvent: function getTimestampAtEvent (e, day, clamp) {
        if ( clamp === void 0 ) clamp = false;

        var timestamp = copyTimestamp(day);
        var bounds = (e.currentTarget).getBoundingClientRect();
        var baseMinutes = this.startMinute;
        var touchEvent = e;
        var mouseEvent = e;
        var touches = touchEvent.changedTouches || touchEvent.touches;
        var clientY = touches && touches[0] ? touches[0].clientY : mouseEvent.clientY;
        var addIntervals = (clientY - bounds.top) / this.parsedIntervalHeight;
        var addMinutes = Math.floor((clamp ? Math.floor(addIntervals) : addIntervals) * this.parsedIntervalMinutes);
        var minutes = baseMinutes + addMinutes;

        return updateMinutes(timestamp, minutes, this.times.now)
      },

      getTimestampAtEventX: function getTimestampAtEventX (e, day, clamp) {
        if ( clamp === void 0 ) clamp = false;

        var timestamp = copyTimestamp(day);
        var bounds = (e.currentTarget).getBoundingClientRect();
        // const baseMinutes = this.startMinute
        var touchEvent = e;
        var mouseEvent = e;
        var touches = touchEvent.changedTouches || touchEvent.touches;
        var clientX = touches && touches[0] ? touches[0].clientX : mouseEvent.clientX;
        var addIntervals = (clientX - bounds.left) / this.parsedIntervalWidth;
        var addMinutes = Math.floor((clamp ? Math.floor(addIntervals) : addIntervals) * this.parsedIntervalMinutes);
        var minutes = addMinutes + (day.hour * 60 + day.minute);

        return updateMinutes(timestamp, minutes, this.times.now)
      },

      getScopeForSlot: function getScopeForSlot (timestamp, idx) {
        var scope = { timestamp: copyTimestamp(timestamp) };
        scope.timeStartPos = this.timeStartPos;
        scope.timeDurationHeight = this.timeDurationHeight;
        if (idx !== undefined) {
          scope.index = idx;
        }
        return scope
      },

      getScopeForSlotX: function getScopeForSlotX (timestamp, idx) {
        var scope = { timestamp: copyTimestamp(timestamp) };
        scope.timeStartPosX = this.timeStartPosX;
        scope.timeDurationWidth = this.timeDurationWidth;
        if (idx !== undefined) {
          scope.index = idx;
        }
        return scope
      },

      scrollToTime: function scrollToTime (time) {
        var y = this.timeStartPos(time);
        var scrollArea = this.$refs.scrollArea;

        if (y === false || !scrollArea) {
          return false
        }

        scrollArea.scrollTop = y;

        return true
      },

      scrollToTimeX: function scrollToTimeX (time) {
        var x = this.timeStartPosX(time);
        var scrollArea = this.$refs.scrollArea;

        if (x === false || !scrollArea) {
          return false
        }

        scrollArea.scrollLeft = x;

        return true
      },

      timeDurationHeight: function timeDurationHeight (minutes) {
        return minutes / this.parsedIntervalMinutes * this.parsedIntervalHeight
      },

      timeDurationWidth: function timeDurationWidth (minutes) {
        return minutes / this.parsedIntervalMinutes * this.parsedIntervalWidth
      },

      heightToMinutes: function heightToMinutes (height) {
        return parseInt(height, 10) * this.parsedIntervalMinutes / this.parsedIntervalHeight
      },

      widthToMinutes: function widthToMinutes (width) {
        return parseInt(width, 10) * this.parsedIntervalMinutes / this.parsedCellWidth
      },

      timeStartPos: function timeStartPos (time, clamp) {
        if ( clamp === void 0 ) clamp = true;

        var minutes = parseTime(time);
        if (minutes === false) { return false }

        var min = this.startMinute;
        var gap = this.parsedIntervalCount * this.parsedIntervalMinutes;
        var delta = (minutes - min) / gap;
        var y = delta * this.bodyHeight;

        if (clamp) {
          if (y < 0) {
            y = 0;
          }
          if (y > this.bodyHeight) {
            y = this.bodyHeight;
          }
        }

        return y
      },

      timeStartPosX: function timeStartPosX (time, clamp) {
        if ( clamp === void 0 ) clamp = true;

        var minutes = parseTime(time);
        if (minutes === false) { return false }

        var min = this.startMinute;
        var gap = this.parsedIntervalCount * this.parsedIntervalMinutes;
        var delta = (minutes - min) / gap;
        var x = delta * this.bodyWidth;

        if (clamp) {
          if (x < 0) {
            x = 0;
          }
          if (x > this.bodyWidth) {
            x = this.bodyWidth;
          }
        }

        return x
      }
    }
  };

  // Quasar

  /* @vue/component */
  var QCalendarDaily = {
    name: 'QCalendarDaily',

    mixins: [
      CalendarIntervals
    ],

    props: {
      direction: {
        type: String,
        default: 'next'
      }
    },

    directives: { Resize: Resize },

    data: function data () {
      return {
        scrollWidth: 0
      }
    },

    computed: {
      computedWidth: function computedWidth () {
        return 100 / this.days.length
      }
    },

    mounted: function mounted () {
      this.init();
    },

    watch: {
      noScroll: function noScroll (val) {
        if (val === true) {
          this.scrollWidth = 0;
        }
        else {
          this.$nextTick(this.onResize);
        }
      }
    },

    methods: {
      init: function init () {
        this.$nextTick(this.onResize);
      },

      onResize: function onResize () {
        this.scrollWidth = this.getScrollWidth();
      },

      getScrollWidth: function getScrollWidth () {
        var area = this.$refs.scrollArea;
        var pane = this.$refs.pane;

        return area && pane ? (area.offsetWidth - pane.offsetWidth) : 0
      },

      __renderHead: function __renderHead (h) {
        var component = h('div', {
          staticClass: 'q-calendar-daily__head',
          style: {
            marginRight: this.scrollWidth + 'px'
          }
        }, [
          this.__renderHeadIntervals(h) ].concat( this.__renderHeadDays(h)
        ));

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }
        return component
      },

      __renderHeadIntervals: function __renderHeadIntervals (h) {
        var this$1$1 = this;

        var slot = this.$scopedSlots['intervals-header'];

        return h('div', {
          staticClass: 'q-calendar-daily__intervals-head q-calendar-daily__intervals-head--text',
          on: this.getDefaultMouseEventHandlers(':interval:header2', function (event) {
            return { scope: { days: this$1$1.days }, event: event }
          })
        }, [
          slot && slot(this.days)
        ])
      },

      __renderHeadDays: function __renderHeadDays (h) {
        var this$1$1 = this;

        if (this.days.length === 1 && this.columnCount !== undefined && parseInt(this.columnCount, 10) > 0) {
          // return [...new Array(parseInt(this.columnCount, 10))]
          return Array.apply(null, new Array(parseInt(this.columnCount, 10)))
            .map(function (_, i) { return i + parseInt(this$1$1.columnIndexStart, 10); })
            .map(function (idx) { return this$1$1.__renderHeadDay(h, this$1$1.days[0], idx); })
        }
        else {
          return this.days.map(function (day) { return this$1$1.__renderHeadDay(h, day); })
        }
      },

      __renderHeadDay: function __renderHeadDay (h, day, idx) {
        var this$1$1 = this;

        var headDaySlot = this.$scopedSlots['head-day'];
        var dayHeaderSlot = this.$scopedSlots['day-header'];
        var activeDate = this.noActiveDate !== true && this.value === day.date;
        var scope = this.getScopeForSlot(day, idx);
        scope.activeDate = activeDate;
        var width = this.computedWidth;
        var dragOver;

        return h('div', {
          key: day.date + (idx !== undefined ? '-' + idx : ''),
          staticClass: 'q-calendar-daily__head-day',
          class: Object.assign({}, this.getRelativeClasses(day),
            {'q-active-date': activeDate,
            'q-calendar-daily__head-day--droppable': dragOver}),
          style: {
            maxWidth: width + '%'
          },
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, day, 'day', idx);
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, day, 'day', idx);
              }
            }
          },
          on: Object.assign({}, this.getDefaultMouseEventHandlers(':day:header2', function (event) {
              return { scope: scope, event: event }
            }))
        }, [
          headDaySlot !== undefined && headDaySlot(scope),
          headDaySlot === undefined && this.columnHeaderBefore === true && this.__renderColumnHeaderBefore(h, day, idx),
          headDaySlot === undefined && this.noDefaultHeaderText !== true && this.__renderHeadWeekday(h, day),
          headDaySlot === undefined && this.noDefaultHeaderBtn !== true && this.__renderHeadDayBtn(h, day),
          headDaySlot === undefined && dayHeaderSlot && dayHeaderSlot(scope),
          headDaySlot === undefined && this.columnHeaderAfter === true && this.__renderColumnHeaderAfter(h, day, idx)
        ])
      },

      __renderHeadWeekday: function __renderHeadWeekday (h, day) {
        var slot = this.$scopedSlots['day-header-label'];
        var scope = this.getScopeForSlot(day);
        scope.shortWeekdayLabel = this.shortWeekdayLabel;

        return h('div', {
          staticClass: 'ellipsis q-calendar-daily__head-weekday'
        }, [
          (slot && slot(scope)) || this.__renderHeadDayLabel(h, day, this.shortWeekdayLabel)
        ])
      },

      __renderHeadDayLabel: function __renderHeadDayLabel (h, day, label) {
        return h('span', {
          staticClass: 'ellipsis'
        }, this.weekdayFormatter(day, label))
      },

      __renderHeadDayBtn: function __renderHeadDayBtn (h, day) {
        var activeDate = this.noActiveDate !== true && this.value === day.date;
        var dayLabel = this.dayFormatter(day, false);
        var dayLabelSlot = this.$scopedSlots['day-label'];
        var dayBtnSlot = this.$scopedSlots['day-btn'];
        var scope = { dayLabel: dayLabel, timestamp: day, activeDate: activeDate };

        return dayBtnSlot
          ? dayBtnSlot(scope)
          : h(quasar.QBtn, {
            staticClass: 'q-calendar-daily__head-day-label',
            props: {
              unelevated: true,
              round: true,
              dense: true,
              noCaps: true,
              outline: day.current === true,
              disable: day.disabled
            },
            on: this.getDefaultMouseEventHandlers(':date2', function (event) {
              return { scope: { timestamp: day }, event: event }
            })
          }, [
            dayLabelSlot ? dayLabelSlot(scope) : dayLabel
          ])
      },

      __renderColumnHeaderBefore: function __renderColumnHeaderBefore (h, day, idx) {
        var slot = this.$scopedSlots['column-header-before'];
        var scope = { timestamp: day };
        scope.index = idx;
        return h('div', {
          staticClass: 'q-calendar-daily__column-header--before'
        }, [
          slot && slot(scope)
        ])
      },

      __renderColumnHeaderAfter: function __renderColumnHeaderAfter (h, day, idx) {
        var slot = this.$scopedSlots['column-header-after'];
        var scope = { timestamp: day };
        scope.index = idx;
        return h('div', {
          staticClass: 'q-calendar-daily__column-header--after'
        }, [
          slot && slot(scope)
        ])
      },

      __renderBody: function __renderBody (h) {
        return h('div', {
          staticClass: 'q-calendar-daily__body'
        }, [
          this.__renderScrollArea(h)
        ])
      },

      __renderScrollArea: function __renderScrollArea (h) {
        if (this.noScroll !== undefined && this.noScroll === true) {
          return this.__renderPane(h)
        }
        else {
          return h('div', {
            ref: 'scrollArea',
            staticClass: 'q-calendar-daily__scroll-area'
          }, [
            this.__renderPane(h)
          ])
        }
      },

      __renderPane: function __renderPane (h) {
        return h('div', {
          ref: 'pane',
          staticClass: 'q-calendar-daily__pane',
          style: {
            height: convertToUnit(this.bodyHeight)
          }
        }, [
          this.__renderDayContainer(h)
        ])
      },

      __renderDayContainer: function __renderDayContainer (h) {
        var slot = this.$scopedSlots['day-container'];

        var component = h('div', {
          staticClass: 'q-calendar-daily__day-container'
        }, [
          this.__renderBodyIntervals(h) ].concat( this.__renderDays(h),
          [slot && slot(this.days)]
        ));

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }
        return component
      },

      __renderDays: function __renderDays (h) {
        var this$1$1 = this;

        if (this.days.length === 1 && this.columnCount && parseInt(this.columnCount, 10) > 0) {
          return Array.apply(null, new Array(parseInt(this.columnCount, 10)))
            .map(function (_, i) { return i + parseInt(this$1$1.columnIndexStart, 10); })
            .map(function (i) { return this$1$1.__renderDay(h, this$1$1.days[0], 0, i); })
        }
        else {
          return this.days.map(function (day, index) { return this$1$1.__renderDay(h, day, index); })
        }
      },

      __renderDay: function __renderDay (h, day, dayIndex, idx) {
        var this$1$1 = this;

        var slot = this.$scopedSlots['day-body'];
        var scope = this.getScopeForSlot(day, idx);
        var width = this.computedWidth;

        return h('div', {
          key: day.date + (idx !== undefined ? ':' + idx : ''),
          staticClass: 'q-calendar-daily__day',
          class: this.getRelativeClasses(day),
          style: {
            maxWidth: width + '%'
          },
          on: Object.assign({}, this.getDefaultMouseEventHandlers(':time2', function (event) {
              var scope = this$1$1.getScopeForSlot(this$1$1.getTimestampAtEvent(event, day, this$1$1.timeClicksClamped), idx);
              return { scope: scope, event: event }
            }))
        }, this.__renderDayIntervals(h, dayIndex, idx).concat( [slot && slot(scope)]
        ))
      },

      __renderDayIntervals: function __renderDayIntervals (h, index, idx) {
        var this$1$1 = this;

        return this.intervals[index].map(function (interval) { return this$1$1.__renderDayInterval(h, interval, idx); })
      },

      __renderDayInterval: function __renderDayInterval (h, interval, idx) {
        var this$1$1 = this;

        var height = convertToUnit(this.intervalHeight);
        var styler = this.intervalStyle || this.intervalStyleDefault;
        var slot = this.$scopedSlots.interval;
        var scope = this.getScopeForSlot(interval, idx);
        var dragOver;

        var data = {
          key: getDateTime(interval),
          staticClass: interval.minute === 0 ? 'q-calendar-daily__day-interval' : 'q-calendar-daily__day-interval--section',
          class: Object.assign({}, this.getIntervalClasses(interval, this.selectedDates, this.selectedStartEndDates),
            {'q-calendar-daily__day-interval--droppable': dragOver}),
          style: Object.assign({}, {height: height},
            styler(interval)),
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, interval, 'interval', idx);
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, interval, 'interval', idx);
              }
            }
          }
        };

        var children = slot ? slot(scope) : undefined;

        return h('div', data, children)
      },

      __renderBodyIntervals: function __renderBodyIntervals (h) {
        var this$1$1 = this;

        var data = {
          staticClass: 'q-calendar-daily__intervals-body',
          on: Object.assign({}, this.getDefaultMouseEventHandlers(':interval2', function (event) {
              var timestamp = this$1$1.getTimestampAtEvent(event, this$1$1.parsedStart, this$1$1.timeClicksClamped);
              return { scope: { timestamp: timestamp }, event: event }
            }))
        };

        return h('div', data, this.__renderIntervalLabels(h))
      },

      __renderIntervalLabels: function __renderIntervalLabels (h) {
        var this$1$1 = this;

        return this.intervals[0].map(function (interval) { return this$1$1.__renderIntervalLabel(h, interval); })
      },

      __renderIntervalLabel: function __renderIntervalLabel (h, interval) {
        var height = convertToUnit(this.intervalHeight);
        var short = this.shortIntervalLabel;
        var shower = this.showIntervalLabel || this.showIntervalLabelDefault;
        var show = shower(interval);
        var label = show ? this.intervalFormatter(interval, short) : undefined;

        return h('div', {
          key: interval.time,
          staticClass: 'q-calendar-daily__interval',
          style: {
            height: height
          }
        }, [
          h('div', {
            staticClass: 'q-calendar-daily__interval-text'
          }, label)
        ])
      }
    },

    render: function render (h) {
      return h('div', {
        class: 'q-calendar-daily',
        directives: [{
          modifiers: { quiet: true },
          name: 'resize',
          value: this.onResize
        }]
      }, [
        !this.hideHeader && this.__renderHead(h),
        this.__renderBody(h)
      ])
    }
  };

  // Mixins

  var CalendarScheduler = {
    name: 'CalendarWithScheduler',

    mixins: [
      CalendarBase
    ],

    props: Object.assign({}, props.scheduler),

    computed: {
      parsedResourceHeight: function parsedResourceHeight () {
        var height = parseFloat(this.resourceHeight);
        if (height === 0) {
          return 'auto'
        }
        return height
      },

      parsedResourceWidth: function parsedResourceWidth () {
        return parseFloat(this.resourceWidth)
      },

      bodyHeight: function bodyHeight () {
        var this$1$1 = this;

        var height = 0;
        var getHeight = function (resources) {
          var resourceHeight = 0;
          resources.forEach(function (resource) {
            resourceHeight += resource.height !== void 0 ? resource.height : this$1$1.parsedResourceHeight;
            if (resource.children && resource.children.length > 0 && resource.expanded === true) {
              resourceHeight += getHeight(resource.children);
            }
          });
          return resourceHeight
        };

        if (this.resources && this.resources.length > 0) {
          height += getHeight(this.resources);
        }

        return height
      },

      days: function days () {
        return createDayList(
          this.parsedStart,
          this.parsedEnd,
          this.times.today,
          this.weekdaySkips,
          this.disabledBefore,
          this.disabledAfter,
          this.disabledWeekdays,
          this.disabledDays,
          this.maxDays
        )
      }
    },

    methods: {
      resourceStyleDefault: function resourceStyleDefault (_timestamp) {
        return undefined
      },

      getTimestampAtEvent: function getTimestampAtEvent (e, day) {
        var timestamp = copyTimestamp(day);
        return updateRelative(timestamp, this.times.now, false)
      },

      getScopeForSlot: function getScopeForSlot (timestamp, idx, resource) {
        var scope = { timestamp: copyTimestamp(timestamp) };
        if (idx !== undefined) {
          scope.index = idx;
        }
        if (resource !== undefined) {
          scope.resource = resource;
        }
        return scope
      }
    }
  };

  // Quasar

  // Icons
  var mdiMenuRight$1 = 'M10,17L15,12L10,7V17Z';
  var mdiMenuDown$1 = 'M7,10L12,15L17,10H7Z';

  /* @vue/component */
  var QCalendarScheduler = {
    name: 'QCalendarScheduler',

    mixins: [
      CalendarScheduler
    ],

    directives: { Resize: Resize },

    props: {
      direction: {
        type: String,
        default: 'next'
      }
    },

    data: function data () {
      return {
        scrollWidth: 0
      }
    },

    created: function created () {
      this.mdiMenuRight = mdiMenuRight$1;
      this.mdiMenuDown = mdiMenuDown$1;
    },

    mounted: function mounted () {
      this.init();
    },

    watch: {
      noScroll: function noScroll (val) {
        if (val === true) {
          this.scrollWidth = 0;
        }
        else {
          this.$nextTick(this.onResize);
        }
      }
    },

    computed: {
      isSticky: function isSticky () {
        return this.cellWidth !== undefined
      }
    },

    methods: {
      init: function init () {
        this.$nextTick(this.onResize);
      },

      onResize: function onResize () {
        this.scrollWidth = this.getScrollWidth();
      },

      getScrollWidth: function getScrollWidth () {
        var area = this.$refs.scrollArea;
        var pane = this.$refs.pane;

        return area && pane ? (area.offsetWidth - pane.offsetWidth) : 0
      },

      resourceStartPos: function resourceStartPos (resource, clamp) {
        if ( clamp === void 0 ) clamp = true;

        var index = this.resource.indexOf(resource);
        var y = index * this.parsedResourceHeight;

        if (clamp) {
          if (y < 0) {
            y = 0;
          }
          if (y > this.bodyHeight) {
            y = this.bodyHeight;
          }
        }

        return y
      },

      __renderHead: function __renderHead (h) {
        var component = h('div', {
          staticClass: 'q-calendar-scheduler__head' + (this.isSticky === true ? ' q-calendar__sticky' : ''),
          style: {
            marginRight: this.getScrollWidth() + 'px'
          }
        }, [
          this.__renderHeadResources(h),
          this.__renderHeadDaysWrapper(h)
        ]);

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }
        return component
      },

      __renderHeadResources: function __renderHeadResources (h) {
        var slot = this.$scopedSlots['scheduler-resources-header'];
        var width = convertToUnit(this.parsedResourceWidth);

        var scope = {
          days: this.days,
          resources: this.resources
        };

        return h('div', {
          staticClass: 'q-calendar-scheduler__resource-head' + (this.isSticky === true ? ' q-calendar__sticky' : ''),
          style: {
            minWidth: width,
            maxWidth: width
          },
          on: this.getDefaultMouseEventHandlers(':resource:header2', function (event) {
            return { scope: scope, event: event }
          })
        }, [
          slot && slot(this.days)
        ])
      },

      __renderHeadDaysWrapper: function __renderHeadDaysWrapper (h) {
        return h('div', {
          staticClass: 'q-calendar-scheduler__head-days-body'
        }, [].concat( this.__renderHeadDays(h) ))
      },

      __renderHeadDays: function __renderHeadDays (h) {
        var this$1$1 = this;

        if (this.days.length === 1 && this.columnCount && parseInt(this.columnCount, 10) > 0) {
          return Array.apply(null, new Array(parseInt(this.columnCount, 10)))
            .map(function (_, i) { return i + parseInt(this$1$1.columnIndexStart, 10); })
            .map(function (idx) { return this$1$1.__renderHeadDay(h, this$1$1.days[0], idx); })
        }
        else {
          return this.days.map(function (day) { return this$1$1.__renderHeadDay(h, day); })
        }
      },

      __renderHeadDay: function __renderHeadDay (h, day, idx) {
        var this$1$1 = this;

        var headDaySlot = this.$scopedSlots['head-day'];
        var dayHeaderSlot = this.$scopedSlots['scheduler-day-header'];
        var activeDate = this.noActiveDate !== true && this.value === day.date;
        var scope = this.getScopeForSlot(day, idx);
        scope.activeDate = activeDate;
        var width = this.isSticky === true ? this.cellWidth : 100 / this.days.length + '%';
        var dragOver;

        return h('div', {
          key: day.date + (idx !== undefined ? ':' + idx : ''),
          staticClass: 'q-calendar-scheduler__head-day',
          class: Object.assign({}, this.getRelativeClasses(day),
            {'q-active-date': activeDate,
            'q-calendar-scheduler__head-day--droppable': dragOver}),
          style: {
            minWidth: width,
            maxWidth: width
          },
          domProps: {
            ondragover: function (_event) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(_event, day, 'day', idx);
              }
            },
            ondrop: function (_event) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(_event, day, 'day', idx);
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':day:header2', function (event) {
            return { scope: scope, event: event }
          })
        }, [
          headDaySlot !== undefined && headDaySlot(scope),
          headDaySlot === undefined && this.columnHeaderBefore === true && this.__renderColumnHeaderBefore(h, day, idx),
          headDaySlot === undefined && this.noDefaultHeaderText !== true && this.__renderHeadWeekday(h, day, idx),
          headDaySlot === undefined && this.noDefaultHeaderBtn !== true && this.__renderHeadDayBtn(h, day, idx),
          headDaySlot === undefined && dayHeaderSlot && dayHeaderSlot(scope),
          headDaySlot === undefined && this.columnHeaderAfter === true && this.__renderColumnHeaderAfter(h, day, idx)
        ])
      },

      __renderHeadWeekday: function __renderHeadWeekday (h, day) {
        return h('div', {
          staticClass: 'ellipsis q-calendar-scheduler__head-weekday'
        }, [
          this.__renderHeadDayLabel(h, day, this.shortWeekdayLabel)
        ])
      },

      __renderHeadDayLabel: function __renderHeadDayLabel (h, day, label) {
        return h('span', {
          staticClass: 'ellipsis'
        }, this.weekdayFormatter(day, label))
      },

      __renderHeadDayBtn: function __renderHeadDayBtn (h, day, idx) {
        var activeDate = this.noActiveDate !== true && this.value === day.date;
        var dayLabel = this.dayFormatter(day, false);
        var dayLabelSlot = this.$scopedSlots['day-label'];
        var dayBtnSlot = this.$scopedSlots['day-btn'];
        var scope = { timestamp: day, index: idx, dayLabel: dayLabel, activeDate: activeDate };

        return dayBtnSlot
          ? dayBtnSlot(scope)
          : h(quasar.QBtn, {
            staticClass: 'q-calendar-scheduler__head-day-label',
            props: {
              unelevated: true,
              round: true,
              dense: true,
              noCaps: true,
              outline: day.current === true,
              disable: day.disabled
            },
            on: this.getDefaultMouseEventHandlers(':date2', function (event) {
              return { scope: { timestamp: day }, event: event }
            })
          }, [
            dayLabelSlot ? dayLabelSlot(scope) : dayLabel
          ])
      },

      __renderColumnHeaderBefore: function __renderColumnHeaderBefore (h, day, idx) {
        var slot = this.$scopedSlots['scheduler-column-header-before'];
        var scope = { timestamp: day };
        scope.index = idx;
        return h('div', {
          staticClass: 'q-calendar-scheduler__column-header--before'
        }, [
          slot && slot(scope)
        ])
      },

      __renderColumnHeaderAfter: function __renderColumnHeaderAfter (h, day, idx) {
        var slot = this.$scopedSlots['scheduler-column-header-after'];
        var scope = { timestamp: day };
        scope.index = idx;
        return h('div', {
          staticClass: 'q-calendar-scheduler__column-header--after'
        }, [
          slot && slot(scope)
        ])
      },

      __renderBody: function __renderBody (h) {
        var component = h('div', {
          staticClass: 'q-calendar-scheduler__body'
        }, [
          this.__renderScrollArea(h)
        ]);

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }
        return component
      },

      __renderScrollArea: function __renderScrollArea (h) {
        if (this.noScroll !== undefined && this.noScroll === true) {
          return this.__renderPane(h)
        }
        else {
          return h('div', {
            ref: 'scrollArea',
            staticClass: 'q-calendar-scheduler__scroll-area'
          }, [
            this.isSticky !== true && this.__renderPane(h),
            this.isSticky === true && this.__renderDayContainer(h)
          ])
        }
      },

      __renderPane: function __renderPane (h) {
        return h('div', {
          ref: 'pane',
          staticClass: 'q-calendar-scheduler__pane',
          style: {
            height: this.resourceHeight === 0 ? 'auto' : convertToUnit(this.bodyHeight)
          }
        }, [
          this.__renderDayContainer(h)
        ])
      },

      __renderDayContainer: function __renderDayContainer (h) {
        return h('div', {
          staticClass: 'q-calendar-scheduler__day-container'
        }, [
          this.isSticky === true && this.__renderHead(h),
          h('div', {
            staticClass: ''
          }, [
            this.__renderResources(h)
          ])
        ])
      },

      __renderResources: function __renderResources (h, resources, indentLevel) {
        var this$1$1 = this;
        if ( resources === void 0 ) resources = undefined;
        if ( indentLevel === void 0 ) indentLevel = 0;

        if (resources === undefined) {
          resources = this.resources;
        }
        return resources.map(function (resource, idx) {
          return this$1$1.__renderResourceRow(h, resource, idx, indentLevel)
        })
      },

      __renderResourceRow: function __renderResourceRow (h, resource, idx, indentLevel) {
        if ( indentLevel === void 0 ) indentLevel = 0;

        var height = resource.height !== void 0 ? convertToUnit(resource.height) : convertToUnit(this.parsedResourceHeight);
        var style = { height: height };
        var resourceRow = h('div', {
          staticClass: 'q-calendar-scheduler__resource-row',
          style: style
        }, [
          this.__renderResource(h, resource, idx, indentLevel),
          this.__renderBodyResources(h, resource, idx, indentLevel)
        ]);
        if (resource.expanded === true) {
          return [resourceRow ].concat( this.__renderResources(h, resource.children, indentLevel + 1))
        }

        return [resourceRow]
      },

      __renderResource: function __renderResource (h, resource, idx, indentLevel) {
        if ( indentLevel === void 0 ) indentLevel = 0;

        return this.__renderResourceLabel(h, resource, idx, indentLevel)
      },

      __renderBodyResources: function __renderBodyResources (h, resource, idx, indentLevel) {

        var width = convertToUnit(this.parsedResourceWidth);

        var data = {
          staticClass: 'q-calendar-scheduler__resources-body',
          style: {
            width: width
          }
        };

        return h('div', data,
          [].concat( this.__renderDays(h, resource, idx) ))
      },

      __renderDays: function __renderDays (h, resource, idx) {
        var this$1$1 = this;

        if (this.days.length === 1 && this.columnCount && parseInt(this.columnCount, 10) > 0) {
          return Array.apply(null, new Array(parseInt(this.columnCount, 10)))
            .map(function (_, i) { return i + parseInt(this$1$1.columnIndexStart, 10); })
            .map(function (i) { return this$1$1.__renderDay(h, resource, this$1$1.days[0], i); })
        }
        else {
          return this.days.map(function (day, index) { return this$1$1.__renderDay(h, resource, day, index, idx); })
        }
      },

      __renderDay: function __renderDay (h, resource, day, idx, resourceIndex) {
        var width = this.isSticky === true ? this.cellWidth : 100 / this.days.length + '%';
        var key = day.date + (resourceIndex !== undefined ? ':' + resourceIndex : '') + (idx !== undefined ? ':' + idx : '');

        return h('div', {
          key: key,
          staticClass: 'q-calendar-scheduler__day',
          class: this.getRelativeClasses(day),
          style: {
            minWidth: width,
            maxWidth: width
          }
        }, [
          this.__renderDayResource(h, resource, day, idx, resourceIndex)
        ])
      },

      __renderDayResource: function __renderDayResource (h, resource, day, idx, resourceIndex) {
        var this$1$1 = this;

        var styler = this.resourceStyle || this.resourceStyleDefault;
        var slot = this.$scopedSlots['scheduler-resource-day'];
        var scope = this.getScopeForSlot(day, idx, resource);
        var key = day.date + (resourceIndex !== undefined ? ':' + resourceIndex : '') + (idx !== undefined ? ':' + idx : '');
        var dragOver;

        var style = styler({ timestamp: day, index: resourceIndex, resource: resource });

        var data = {
          key: key,
          staticClass: 'q-calendar-scheduler__day-resource',
          class: {
            'q-calendar-scheduler__day-resource--droppable': dragOver
          },
          style: style,
          domProps: {
            ondragover: function (_event) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(_event, resource, 'resource', resourceIndex);
              }
            },
            ondrop: function (_event) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(_event, resource, 'resource', resourceIndex);
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':resource:day2', function (event) {
            var scope = this$1$1.getScopeForSlot(this$1$1.getTimestampAtEvent(event, day, this$1$1.timeClicksClamped), resourceIndex, resource);
            return { scope: scope, event: event }
          })
        };

        var children = slot ? slot(scope) : undefined;

        return h('div', data, children)
      },

      __renderResourceLabel: function __renderResourceLabel (h, resource, idx, indentLevel) {
        var this$1$1 = this;
        if ( indentLevel === void 0 ) indentLevel = 0;

        var slot = this.$scopedSlots['scheduler-resource'];
        var scope = {
          resource: resource,
          index: idx,
          days: this.days
        };
        var width = convertToUnit(this.parsedResourceWidth);
        var label = resource[this.resourceKey];
        if (label === undefined) {
          /* eslint-disable-next-line */
          console.warn('QCalendarScheduler: resource object requires "resource-key" property to contain resource object key');
        }

        return h('div', {
          key: label + (idx !== undefined ? ':' + idx : ''),
          staticClass: 'q-calendar-scheduler__resource' + (this.isSticky === true ? ' q-calendar__sticky' : ''),
          style: {
            maxWidth: width,
            minWidth: width,
            height: '100%',
            paddingLeft: (10 * indentLevel + 2) + 'px'
          },
          on: this.getDefaultMouseEventHandlers(':resource2', function (event) {
            return { scope: scope, event: event }
          })
        }, [
          slot
            ? slot(scope)
            : h('div', {
              staticClass: 'q-calendar-scheduler__resource-text'
            }, [
              resource.children && resource.children.length > 0 && h(quasar.QIcon, {
                props: {
                  name: (resource.expanded === true ? this.mdiMenuDown : this.mdiMenuRight),
                  size: 'md'
                },
                on: {
                  click: function (e) {
                    resource.expanded = !resource.expanded;
                    e.stopPropagation();
                    this$1$1.$emit('expanded', resource);
                  }
                }
              }),
              label
            ])
        ])
      },

      __renderResourcesError: function __renderResourcesError (h) {
        return h('div', {}, 'No resources have been defined')
      }
    },

    render: function render (h) {
      return h('div', {
        class: 'q-calendar-scheduler',
        directives: [{
          modifiers: { quiet: true },
          name: 'resize',
          value: this.onResize
        }]
      }, [
        this.isSticky !== true && this.hideHeader !== true && this.resources !== undefined && this.__renderHead(h),
        this.resources !== undefined && this.__renderBody(h),
        this.resources === undefined && this.__renderResourcesError(h)
      ])
    }
  };

  // Mixins

  var QCalendarAgenda = {
    name: 'QCalendarAgenda',

    mixins: [
      QCalendarDaily
    ],

    props: Object.assign({}, props.agenda),

    computed: {
      computedColumnCount: function computedColumnCount () {
        return this.days.length +
          (this.leftColumnOptionsValid === true ? this.leftColumnOptions.length : 0) +
          (this.rightColumnOptionsValid === true ? this.rightColumnOptions.length : 0)
      },

      computedWidth: function computedWidth () {
        return 100 / this.computedColumnCount
      },

      leftColumnOptionsValid: function leftColumnOptionsValid () {
        return this.leftColumnOptions !== undefined && Array.isArray(this.leftColumnOptions)
      },

      rightColumnOptionsValid: function rightColumnOptionsValid () {
        return this.rightColumnOptions !== undefined && Array.isArray(this.rightColumnOptions)
      }
    },

    methods: {
      __renderHead: function __renderHead (h) {
        var component = h('div', {
          staticClass: 'q-calendar-daily__head',
          style: {
            marginRight: this.scrollWidth + 'px'
          }
        }, this.__renderHeadDays(h));

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }
        return component
      },

      __renderHeadDays: function __renderHeadDays (h) {
        var this$1$1 = this;

        if (this.days.length === 1 && this.columnCount !== undefined && parseInt(this.columnCount, 10) > 0) {
          // return [...new Array(parseInt(this.columnCount, 10))]
          return [
            Array.apply(null, new Array(parseInt(this.columnCount, 10)))
              .map(function (_, i) { return i + parseInt(this$1$1.columnIndexStart, 10); })
              .map(function (idx) { return this$1$1.__renderHeadDay(h, this$1$1.days[0], idx); })
          ]
        }
        else {
          return [
            this.leftColumnOptionsValid === true && this.leftColumnOptions.map(function (column, _i) { return this$1$1.__renderColumnHead(h, column, _i); }),
            this.days.map(function (day) { return this$1$1.__renderHeadDay(h, day); }),
            this.rightColumnOptionsValid === true && this.rightColumnOptions.map(function (column, _i) { return this$1$1.__renderColumnHead(h, column, _i); })
          ]
        }
      },

      __renderPane: function __renderPane (h) {
        return h('div', {
          ref: 'pane',
          staticClass: 'q-calendar-daily__pane',
          style: {
            // height: convertToUnit(this.bodyHeight)
          }
        }, [
          this.__renderDayContainer(h)
        ])
      },

      __renderBodyIntervals: function __renderBodyIntervals (h) {
        return undefined
      },

      __renderColumnHead: function __renderColumnHead (h, column, idx) {
        var this$1$1 = this;

        var slot = this.$scopedSlots['column-header'];
        var scope = column;
        var width = this.computedWidth;
        var dragOver;

        return h('div', {
          key: (this.columnOptionsId !== undefined ? column[this.columnOptionsId] : undefined),
          staticClass: 'q-calendar-daily__head-day',
          class: {
            'q-column-day': true,
            'q-calendar-daily__head-day--droppable': dragOver
          },
          style: {
            maxWidth: width + '%'
          },
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, column, 'column', idx);
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, column, 'column', idx);
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':column:header2', function (event, eventName) {
            return { scope: { column: column, index: idx }, event: event }
          })
        }, [
          this.noDefaultHeaderText !== true && this.__renderHeadColumn(h, column),
          slot && slot(scope)
        ])
      },

      __renderHeadColumn: function __renderHeadColumn (h, column) {
        var slot = this.$scopedSlots['column-header-label'];
        var scope = column;

        return h('div', {
          staticClass: 'ellipsis q-calendar-daily__head-weekday'
        }, [
          slot && slot(scope),
          !slot && this.__renderHeadColumnLabel(h, (this.columnOptionsLabel !== undefined ? column[this.columnOptionsLabel] : column.label))
        ])
      },

      __renderHeadColumnLabel: function __renderHeadColumnLabel (h, label) {
        return h('span', {
          staticClass: 'ellipsis'
        }, label)
      },

      __renderDays: function __renderDays (h) {
        var this$1$1 = this;

        if (this.days.length === 1 && this.columnCount && parseInt(this.columnCount, 10) > 0) {
          return [
            Array.apply(null, new Array(parseInt(this.columnCount, 10)))
              .map(function (_, i) { return i + parseInt(this$1$1.columnIndexStart, 10); })
              .map(function (i) { return this$1$1.__renderDay(h, this$1$1.days[0], 0, i); })
          ]
        }
        else {
          return [
            this.leftColumnOptionsValid === true && this.leftColumnOptions.map(function (column, _i) { return this$1$1.__renderColumn(h, column, _i); }),
            this.days.map(function (day, index) { return this$1$1.__renderDay(h, day, index); }),
            this.rightColumnOptionsValid === true && this.rightColumnOptions.map(function (column, _i) { return this$1$1.__renderColumn(h, column, _i); })
          ]
        }
      },

      __renderColumn: function __renderColumn (h, column, idx) {
        var this$1$1 = this;

        var slot = this.$scopedSlots['column-body'];
        var scope = { column: column, index: idx };
        var width = this.computedWidth;
        var dragOver;

        return h('div', {
          key: (this.columnOptionsId !== undefined ? column[this.columnOptionsId] : undefined),
          staticClass: 'q-calendar-daily__day',
          class: {
            'q-column-day': true,
            'q-calendar-daily__day--droppable': dragOver
          },
          style: {
            maxWidth: width + '%'
          },
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, column, 'column');
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, column, 'column');
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':column2', function (event, eventName) {
            if (eventName.indexOf('2') > -1) {
              return { scope: scope, event: event }
            }
            else {
              return { scope: scope, event: event }
            }
          })
        }, [
          slot && slot(scope)
        ])
      },

      __renderDay: function __renderDay (h, day, dayIndex, idx) {
        var this$1$1 = this;

        var slot = this.$scopedSlots['day-body'];
        var scope = { timestamp: day };
        var width = this.computedWidth;
        var dragOver;

        return h('div', {
          key: day.date + (idx !== undefined ? ':' + idx : ''),
          staticClass: 'q-calendar-daily__day',
          class: Object.assign({}, this.getRelativeClasses(day),
            {'q-calendar-daily__day--droppable': dragOver}),
          style: {
            maxWidth: width + '%'
          },
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, day, 'day');
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, day, 'day');
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':time2', function (event, eventName) {
            var scope = this$1$1.getScopeForSlot(this$1$1.getTimestampAtEvent(event, day, this$1$1.timeClicksClamped), idx);
            scope.index = idx;
            return { scope: scope, event: event }
          })
        }, [
          slot && slot(scope)
        ])
      }
    },

    render: function render (h) {
      return h('div', {
        class: 'q-calendar-agenda'
      }, [
        !this.hideHeader && this.__renderHead(h),
        this.__renderBody(h)
      ])
    }
  };

  // Quasar

  // Icons
  var mdiMenuRight = 'M10,17L15,12L10,7V17Z';
  var mdiMenuDown = 'M7,10L12,15L17,10H7Z';

  /* @vue/component */
  var QCalendarResource = {
    name: 'QCalendarResource',

    mixins: [
      CalendarIntervals
    ],

    props: Object.assign({}, props.intervals,
      props.resource,
      {direction: {
        type: String,
        default: 'next'
      }}),

    data: function data () {
      return {
        minWidth: '100px'
      }
    },

    created: function created () {
      this.mdiMenuRight = mdiMenuRight;
      this.mdiMenuDown = mdiMenuDown;
    },

    computed: {
      parsedResourceHeight: function parsedResourceHeight () {
        var height = parseFloat(this.resourceHeight);
        if (height === 0) {
          return 'auto'
        }
        return height
      },

      parsedResourceWidth: function parsedResourceWidth () {
        return parseFloat(this.resourceWidth)
      },

      parsedIntervalHeight: function parsedIntervalHeight () {
        var height = parseFloat(this.intervalHeight);
        if (height === 0) {
          return 'auto'
        }
        return height
      }
    },

    methods: {
      __renderHeadIntervals: function __renderHeadIntervals (h) {
        var this$1$1 = this;

        return h('div', {
          staticClass: 'q-calendar-resource__head-intervals'
        }, [
          this.intervals.map(function (intervals) { return intervals.map(function (interval, index) { return this$1$1.__renderHeadInterval(h, interval, index); }); })
        ])
      },

      __renderHeadInterval: function __renderHeadInterval (h, interval, index) {
        var this$1$1 = this;

        var width = convertToUnit(this.parsedIntervalWidth);
        var height = convertToUnit(this.parsedIntervalHeight);
        var slot = this.$scopedSlots['interval-label'];
        var short = this.shortIntervalLabel;
        var label = this.intervalFormatter(interval, short);
        var scope = {
          interval: interval,
          index: index,
          label: label
        };
        var dragOver;

        return slot ? slot(scope) : h('div', {
          staticClass: 'q-calendar-resource__head-label',
          class: {
            'q-calendar-resource__head-label--droppable': dragOver
          },
          style: {
            maxWidth: width,
            minWidth: width,
            height: height
          },
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, interval, 'interval', index);
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, interval, 'interval', index);
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':interval2', function (event, eventName) {
            return { scope: { timestamp: interval, index: index, label: label }, event: event }
          })
        }, label)
      },

      __renderBody: function __renderBody (h) {
        return h('div', {
          staticClass: 'q-calendar-resource__body'
        }, [
          this.__renderScrollArea(h)
        ])
      },

      __renderScrollArea: function __renderScrollArea (h) {
        return h('div', {
          ref: 'scrollArea',
          staticClass: 'q-calendar-resource__scroll-area'
        }, [
          this.__renderDayContainer(h)
        ])
      },

      __renderHead: function __renderHead (h) {
        return h('div', {
          staticClass: 'q-calendar-resource__head' + (this.sticky === true ? ' q-calendar__sticky' : '')
        }, [
          this.__renderHeadResource(h),
          this.__renderHeadIntervals(h)
        ])
      },

      __renderHeadResource: function __renderHeadResource (h) {
        var slot = this.$scopedSlots['resource-header'];
        var width = convertToUnit(this.parsedResourceWidth);
        var height = convertToUnit(this.parsedIntervalHeight);

        var scope = {
          timestamp: this.days[0],
          resources: this.resources,
          intervals: this.intervals[0]
        };
        var intervals = this.intervals;

        return h('div', {
          staticClass: 'q-calendar-resource__resource-head' + (this.sticky === true ? ' q-calendar__sticky' : ''),
          style: {
            maxWidth: width,
            minWidth: width,
            height: height
          },
          on: this.getDefaultMouseEventHandlers(':resource:header2', function (event) {
            return { scope: scope, event: event }
          })
        }, [
          slot && slot({ date: this.value, intervals: intervals })
        ])
      },

      __renderDayContainer: function __renderDayContainer (h) {
        var component = h('div', {
          staticClass: 'q-calendar-resource__day-container'
        }, [
          this.__renderHead(h),
          this.resources === undefined && this.__renderResourcesError(h),
          this.resources !== undefined && this.__renderBodyResources(h)
        ]);

        if (this.animated === true) {
          var transition = 'q-transition--' + (this.direction === 'prev' ? this.transitionPrev : this.transitionNext);
          return h('transition', {
            props: {
              name: transition,
              appear: true
            }
          }, [
            component
          ])
        }
        return component
      },

      __renderBodyResources: function __renderBodyResources (h) {
        var data = {
          staticClass: 'q-calendar-resource__resources-body'
        };

        return h('div', data, this.__renderResources(h))
      },

      __renderResources: function __renderResources (h, resources, indentLevel) {
        var this$1$1 = this;
        if ( resources === void 0 ) resources = undefined;
        if ( indentLevel === void 0 ) indentLevel = 0;

        if (resources === undefined) {
          resources = this.resources;
        }
        return resources.map(function (resource, idx) {
          return this$1$1.__renderResourceRow(h, resource, idx, indentLevel)
        })
      },

      __renderResourceRow: function __renderResourceRow (h, resource, idx, indentLevel) {
        if ( indentLevel === void 0 ) indentLevel = 0;

        var slot = this.$scopedSlots['resource-row'];
        var resourceRow = h('div', {
          staticClass: 'q-calendar-resource__resource-row'
        }, [
          this.__renderResourceLabel(h, resource, idx, indentLevel),
          this.__renderResourceIntervals(h, resource, idx),
          slot && slot({ resource: resource, index: idx })
        ]);
        if (resource.expanded === true) {
          return [resourceRow ].concat( this.__renderResources(h, resource.children, indentLevel + 1))
        }

        return [resourceRow]
      },

      __renderResourceLabel: function __renderResourceLabel (h, resource, idx, indentLevel) {
        var this$1$1 = this;
        if ( indentLevel === void 0 ) indentLevel = 0;

        var slot = this.$scopedSlots['resource-label'];
        var scope = {
          resource: resource,
          index: idx
        };
        var width = convertToUnit(this.parsedResourceWidth);
        var height = resource.height !== void 0 ? convertToUnit(resource.height) : convertToUnit(this.parsedResourceHeight);

        return h('div', {
          key: resource[this.resourceKey] + '-' + idx,
          staticClass: 'q-calendar-resource__resource' + (this.sticky === true ? ' q-calendar__sticky' : ''),
          style: {
            maxWidth: width,
            minWidth: width,
            height: height
          },
          on: this.getDefaultMouseEventHandlers(':resource2', function (event, eventName) {
            return { scope: { resource: resource, index: idx, intervals: this$1$1.intervals }, event: event }
          })
        }, [
          slot ? slot(scope) : this.__renderResourceText(h, resource, idx, indentLevel)
        ])
      },

      __renderResourceText: function __renderResourceText (h, resource, idx, indentLevel) {
        var this$1$1 = this;
        if ( indentLevel === void 0 ) indentLevel = 0;

        var label = resource[this.resourceKey];
        if (label === undefined) {
          /* eslint-disable-next-line */
          console.warn('QCalendarResource: resource object requires "resource-key" property to contain resource object key');
        }

        return h('div', {
          staticClass: 'q-calendar-resource__resource-text',
          style: {
            paddingLeft: (10 * indentLevel) + 'px'
          }
        }, [
          resource.children && resource.children.length > 0 && h(quasar.QIcon, {
            props: {
              name: (resource.expanded === true ? this.mdiMenuDown : this.mdiMenuRight),
              size: 'md'
            },
            on: {
              click: function (e) {
                resource.expanded = !resource.expanded;
                e.stopPropagation();
                this$1$1.$emit('expanded', resource);
              }
            }
          }),
          label
        ])
      },

      __renderResourceIntervals: function __renderResourceIntervals (h, resource, idx) {
        var this$1$1 = this;

        var slot = this.$scopedSlots['resource-intervals'];
        var timeStartPosX = this.timeStartPosX,
          timeDurationWidth = this.timeDurationWidth,
          intervals = this.intervals;
        return h('div', {
          staticClass: 'q-calendar-resource__resource-intervals'
        }, [
          this.intervals.map(function (intervals) { return intervals.map(function (interval) { return this$1$1.__renderResourceInterval(h, resource, interval, idx); }); }),
          slot && slot({ resource: resource, intervals: intervals, timeStartPosX: timeStartPosX, timeDurationWidth: timeDurationWidth })
        ])
      },

      // interval related to resource
      __renderResourceInterval: function __renderResourceInterval (h, resource, interval, idx) {
        var this$1$1 = this;

        // called for each interval
        var slot = this.$scopedSlots['resource-interval'];
        var slotData = { resource: resource, interval: interval };
        var width = convertToUnit(this.parsedIntervalWidth);
        var height = resource.height !== void 0 ? convertToUnit(resource.height) : convertToUnit(this.parsedResourceHeight);
        var dragOver;

        return h('div', {
          staticClass: 'q-calendar-resource__resource-interval',
          class: {
            'q-calendar-resource__resource-interval--droppable': dragOver
          },
          style: {
            maxWidth: width,
            minWidth: width,
            height: height
          },
          domProps: {
            ondragover: function (e) {
              if (this$1$1.dragOverFunc !== undefined) {
                dragOver = this$1$1.dragOverFunc(e, interval, 'resource', resource);
              }
            },
            ondrop: function (e) {
              if (this$1$1.dropFunc !== undefined) {
                this$1$1.dropFunc(e, interval, 'resource', resource);
              }
            }
          },
          on: this.getDefaultMouseEventHandlers(':time2', function (event, eventName) {
            var scope = this$1$1.getScopeForSlotX(this$1$1.getTimestampAtEventX(event, interval, this$1$1.timeClicksClamped));
            scope.resource = resource;
            scope.index = idx;
            return { scope: scope, event: event }
          })
        }, [
          slot && slot(slotData)
        ])
      },

      __renderResourcesError: function __renderResourcesError (h) {
        return h('div', {}, 'No resources have been defined')
      }
    },

    render: function render (h) {
      return h('div', {
        class: 'q-calendar-resource'
      }, [
        this.__renderBody(h)
      ])
    }
  };

  // Mixins

  /* @vue/component */
  var QCalendar = {
    name: 'QCalendar',

    mixins: [
      CalendarBase
    ],

    props: Object.assign({}, props.base,
      props.calendar,
      props.weeks,
      props.intervals,
      props.scheduler,
      props.resource,
      props.agenda),

    data: function () { return ({
      lastStart: undefined,
      lastEnd: undefined,
      emittedValue: ''
    }); },

    computed: {
      __parsedValue: function __parsedValue () {
        return parseTimestamp(this.value) ||
          this.parsedStart ||
          this.times.today
      },

      __renderProps: function __renderProps () {
        var around = this.__parsedValue;
        var component = 'div';
        var maxDays = this.maxDays;
        var start = around;
        var end = around;
        switch (this.view) {
          case 'month':
            component = QCalendarMonthly;
            start = getStartOfMonth(around);
            end = getEndOfMonth(around);
            break
          case 'week':
          case 'week-agenda':
          case 'week-scheduler':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            start = this.getStartOfWeek(around);
            end = this.getEndOfWeek(start);
            maxDays = this.weekdays.length;
            break
          case 'day':
          case 'day-scheduler':
          case 'day-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            maxDays = 1;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
          case '2day':
          case '2day-scheduler':
          case '2day-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            maxDays = 2;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
          case '3day':
          case '3day-scheduler':
          case '3day-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            maxDays = 3;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
          case '4day':
          case '4day-scheduler':
          case '4day-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            maxDays = 4;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
          case '5day':
          case '5day-scheduler':
          case '5day-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            maxDays = 5;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
          case '6day':
          case '6day-scheduler':
          case '6day-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            maxDays = 6;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
          case 'custom-interval':
          case 'custom-scheduler':
          case 'custom-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            end = relativeDays(copyTimestamp(end), nextDay, this.maxDays, this.weekdays);
            updateFormatted(end);
            break
          case 'scheduler':
          case 'agenda':
            component = this.view === 'scheduler' ? QCalendarScheduler : QCalendarAgenda;
            end = relativeDays(copyTimestamp(end), nextDay, this.maxDays, this.weekdays);
            updateFormatted(end);
            break
          case 'month-interval':
          case 'month-scheduler':
          case 'month-agenda':
            component = this.view.endsWith('-agenda') ? QCalendarAgenda : this.view.endsWith('-scheduler') ? QCalendarScheduler : QCalendarDaily;
            start = getStartOfMonth(around);
            end = getEndOfMonth(around);
            updateFormatted(end);
            maxDays = DAYS_IN_MONTH_MAX;
            break
          case 'resource':
          case 'day-resource':
            component = QCalendarResource;
            maxDays = 1;
            end = relativeDays(copyTimestamp(end), nextDay, maxDays, this.weekdays);
            updateFormatted(end);
            break
        }
        return { component: component, start: start, end: end, maxDays: maxDays }
      },

      __isMiniMode: function __isMiniMode () {
        return this.miniMode === true ||
          (this.miniMode === 'auto' && this.breakpoint !== void 0 && this.$q.screen.lt[this.breakpoint])
      }
    },

    beforeMount: function beforeMount () {
      this.emittedValue = this.value;

      // get start and end dates
      this.__checkChange();
    },

    watch: {
      __renderProps: '__checkChange',

      emittedValue: function emittedValue (val, oldVal) {
        this.$emit('input', val);
      }
    },

    methods: {
      __checkChange: function __checkChange () {
        var ref = this.__renderProps;
        var start = ref.start;
        var end = ref.end;
        this.keyValue = 0;
        if (this.lastStart === undefined || this.lastEnd === undefined || start.date !== this.lastStart || end.date !== this.lastEnd) {
          this.lastStart = start.date;
          this.lastEnd = end.date;
          this.$emit('change', { start: start, end: end });
        }
        this.keyValue = getDayIdentifier(start);
      },

      move: function move (amount) {
        if ( amount === void 0 ) amount = 1;

        if (amount === 0) {
          this.moveToToday();
          return
        }
        var moved = copyTimestamp(this.__parsedValue);
        var forward = amount > 0;
        var mover = forward ? nextDay : prevDay;
        var limit = forward ? DAYS_IN_MONTH_MAX : DAY_MIN;
        var times = forward ? amount : -amount;
        this.direction = forward ? 'next' : 'prev';
        var maxDays = this.maxDays;
        var dayCount = this.weekdaySkips.filter(function (x) { return x !== 0; }).length;

        while (--times >= 0) {
          switch (this.view) {
            case 'month':
              moved.day = limit;
              mover(moved);
              break
            case 'week':
            case 'week-scheduler':
            case 'week-agenda':
            case 'scheduler':
              relativeDays(moved, mover, dayCount, this.weekdays);
              break
            case 'day':
            case 'day-scheduler':
            case 'day-agenda':
              maxDays = 1;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
            case '2day':
            case '2day-scheduler':
            case '2day-agenda':
              maxDays = 2;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
            case '3day':
            case '3day-scheduler':
            case '3day-agenda':
              maxDays = 3;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
            case '4day':
            case '4day-scheduler':
            case '4day-agenda':
              maxDays = 4;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
            case '5day':
            case '5day-scheduler':
            case '5day-agenda':
              maxDays = 5;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
            case '6day':
            case '6day-scheduler':
            case '6day-agenda':
              maxDays = 6;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
            case 'custom-interval':
            case 'custom-scheduler':
            case 'custom-agenda':
            case 'agenda':
              relativeDays(moved, mover, this.maxDays, this.weekdays);
              break
            case 'month-interval':
            case 'month-agenda':
            case 'month-scheduler':
              moved.day = limit;
              mover(moved);
              break
            case 'resource':
            case 'day-resource':
              maxDays = 1;
              relativeDays(moved, mover, maxDays, this.weekdays);
              break
          }
        }

        updateWeekday(moved);
        updateFormatted(moved);
        updateDayOfYear(moved);
        updateRelative(moved, this.times.now);

        this.emittedValue = moved.date;
        this.$emit('moved', moved);
      },

      moveToToday: function moveToToday () {
        this.emittedValue = today();
      },

      next: function next (amount) {
        if ( amount === void 0 ) amount = 1;

        this.move(amount);
      },

      prev: function prev (amount) {
        if ( amount === void 0 ) amount = 1;

        this.move(-amount);
      },

      timeStartPos: function timeStartPos (time, clamp) {
        if ( clamp === void 0 ) clamp = true;

        var c = this.$children[0];
        if (c && c.timeStartPos) {
          return c.timeStartPos(time, clamp)
        }
        else {
          return false
        }
      },

      timeStartPosX: function timeStartPosX (time, clamp) {
        if ( clamp === void 0 ) clamp = true;

        var c = this.$children[0];
        if (c && c.timeStartPosX) {
          return c.timeStartPosX(time, clamp)
        }
        else {
          return false
        }
      },

      timeDurationHeight: function timeDurationHeight (minutes) {
        var c = this.$children[0];
        if (c && c.timeDurationHeight) {
          return c.timeDurationHeight(minutes)
        }
        else {
          return -1
        }
      },

      timeDurationWidth: function timeDurationWidth (minutes) {
        var c = this.$children[0];
        if (c && c.timeDurationWidth) {
          return c.timeDurationWidth(minutes)
        }
        else {
          return -1
        }
      },

      heightToMinutes: function heightToMinutes (height) {
        var c = this.$children[0];
        if (c && c.heightToMinutes) {
          return c.heightToMinutes(height)
        }
        else {
          return -1
        }
      },

      widthToMinutes: function widthToMinutes (width) {
        var c = this.$children[0];
        if (c && c.widthToMinutes) {
          return c.widthToMinutes(width)
        }
        else {
          return -1
        }
      },

      scrollToTime: function scrollToTime (time) {
        var c = this.$children[0];
        if (c && c.scrollToTime) {
          return c.scrollToTime(time)
        }
        else {
          return false
        }
      },

      scrollToTimeX: function scrollToTimeX (time) {
        var c = this.$children[0];
        if (c && c.scrollToTimeX) {
          return c.scrollToTimeX(time)
        }
        else {
          return false
        }
      },

      __renderComponent: function __renderComponent (h, component, data) {
        return h(component, data)
      }
    },

    render: function render (h) {
      var this$1$1 = this;

      var ref = this.__renderProps;
      var start = ref.start;
      var end = ref.end;
      var maxDays = ref.maxDays;
      var component = ref.component;

      this.keyValue = getDayIdentifier(start);

      var data = {
        staticClass: (this.__isMiniMode === true ? 'q-calendar-mini' : ''),
        class: {
          'q-calendar__bordered': this.bordered
        },
        key: this.keyValue,
        props: Object.assign({}, this.$props,
          {direction: this.direction,
          start: start.date,
          end: end.date,
          maxDays: maxDays}),
        on: Object.assign({}, this.$listeners,
          {'click:date2': function (ref) {
            var scope = ref.scope;
            var event = ref.event;

            if (this$1$1.$listeners.input !== undefined) {
              if (scope.timestamp.date !== undefined && this$1$1.emittedValue !== scope.timestamp.date) {
                this$1$1.emittedValue = scope.timestamp.date;
              }
            }
            // Because we highjack this event for input, pass it on to parent
            if (this$1$1.$listeners['click:date2']) {
              /* eslint-disable-next-line */
              this$1$1.$emit('click:date2', { scope: scope, event: event });
            }
          }}),
        scopedSlots: this.$scopedSlots
      };

      return h('div', {
        class: {
          'q-calendar--dark': this.dark === true,
          'q-calendar': true
        }
      }, [
        this.__renderComponent(h, component, data)
      ])
    }
  };

  var Plugin = {
    version: version,
    QCalendar: QCalendar,
    // timestamp
    PARSE_REGEX: PARSE_REGEX,
    PARSE_TIME: PARSE_TIME,
    DAYS_IN_MONTH: DAYS_IN_MONTH,
    DAYS_IN_MONTH_LEAP: DAYS_IN_MONTH_LEAP,
    DAYS_IN_MONTH_MIN: DAYS_IN_MONTH_MIN,
    DAYS_IN_MONTH_MAX: DAYS_IN_MONTH_MAX,
    MONTH_MAX: MONTH_MAX,
    MONTH_MIN: MONTH_MIN,
    DAY_MIN: DAY_MIN,
    DAYS_IN_WEEK: DAYS_IN_WEEK,
    MINUTES_IN_HOUR: MINUTES_IN_HOUR,
    HOURS_IN_DAY: HOURS_IN_DAY,
    FIRST_HOUR: FIRST_HOUR,
    MILLISECONDS_IN_MINUTE: MILLISECONDS_IN_MINUTE,
    MILLISECONDS_IN_HOUR: MILLISECONDS_IN_HOUR,
    MILLISECONDS_IN_DAY: MILLISECONDS_IN_DAY,
    MILLISECONDS_IN_WEEK: MILLISECONDS_IN_WEEK,
    Timestamp: Timestamp,
    TimeObject: TimeObject,
    today: today,
    getStartOfWeek: getStartOfWeek,
    getEndOfWeek: getEndOfWeek,
    getStartOfMonth: getStartOfMonth,
    getEndOfMonth: getEndOfMonth,
    parseTime: parseTime,
    validateTimestamp: validateTimestamp,
    parsed: parsed,
    parseTimestamp: parseTimestamp,
    parseDate: parseDate,
    getDayIdentifier: getDayIdentifier,
    getTimeIdentifier: getTimeIdentifier,
    getDayTimeIdentifier: getDayTimeIdentifier,
    diffTimestamp: diffTimestamp,
    updateRelative: updateRelative,
    updateMinutes: updateMinutes,
    updateWeekday: updateWeekday,
    updateDayOfYear: updateDayOfYear,
    updateWorkWeek: updateWorkWeek,
    updateDisabled: updateDisabled,
    updateFormatted: updateFormatted,
    getDayOfYear: getDayOfYear,
    getWorkWeek: getWorkWeek,
    getWeekday: getWeekday,
    isLeapYear: isLeapYear,
    daysInMonth: daysInMonth,
    copyTimestamp: copyTimestamp,
    padNumber: padNumber,
    getDate: getDate,
    getTime: getTime,
    getDateTime: getDateTime,
    nextDay: nextDay,
    prevDay: prevDay,
    relativeDays: relativeDays,
    moveRelativeDays: moveRelativeDays,
    findWeekday: findWeekday,
    getWeekdaySkips: getWeekdaySkips,
    createDayList: createDayList,
    createIntervalList: createIntervalList,
    createNativeLocaleFormatter: createNativeLocaleFormatter,
    makeDate: makeDate,
    makeDateTime: makeDateTime,
    validateNumber: validateNumber,
    isBetweenDates: isBetweenDates,
    isOverlappingDates: isOverlappingDates,
    daysBetween: daysBetween,
    weeksBetween: weeksBetween,
    addToDate: addToDate,
    compareTimestamps: compareTimestamps,
    compareDate: compareDate,
    compareTime: compareTime,
    compareDateTime: compareDateTime,
    // helpers
    convertToUnit: convertToUnit,
    indexOf: indexOf,

    install: function install (Vue) {
      Vue.component(QCalendar.name, QCalendar);
    }
  };

  Vue__default['default'].use(Plugin);

  exports.DAYS_IN_MONTH = DAYS_IN_MONTH;
  exports.DAYS_IN_MONTH_LEAP = DAYS_IN_MONTH_LEAP;
  exports.DAYS_IN_MONTH_MAX = DAYS_IN_MONTH_MAX;
  exports.DAYS_IN_MONTH_MIN = DAYS_IN_MONTH_MIN;
  exports.DAYS_IN_WEEK = DAYS_IN_WEEK;
  exports.DAY_MIN = DAY_MIN;
  exports.FIRST_HOUR = FIRST_HOUR;
  exports.HOURS_IN_DAY = HOURS_IN_DAY;
  exports.MILLISECONDS_IN_DAY = MILLISECONDS_IN_DAY;
  exports.MILLISECONDS_IN_HOUR = MILLISECONDS_IN_HOUR;
  exports.MILLISECONDS_IN_MINUTE = MILLISECONDS_IN_MINUTE;
  exports.MILLISECONDS_IN_WEEK = MILLISECONDS_IN_WEEK;
  exports.MINUTES_IN_HOUR = MINUTES_IN_HOUR;
  exports.MONTH_MAX = MONTH_MAX;
  exports.MONTH_MIN = MONTH_MIN;
  exports.PARSE_REGEX = PARSE_REGEX;
  exports.PARSE_TIME = PARSE_TIME;
  exports.QCalendar = QCalendar;
  exports.TimeObject = TimeObject;
  exports.Timestamp = Timestamp;
  exports.addToDate = addToDate;
  exports.compareDate = compareDate;
  exports.compareDateTime = compareDateTime;
  exports.compareTime = compareTime;
  exports.compareTimestamps = compareTimestamps;
  exports.convertToUnit = convertToUnit;
  exports.copyTimestamp = copyTimestamp;
  exports.createDayList = createDayList;
  exports.createIntervalList = createIntervalList;
  exports.createNativeLocaleFormatter = createNativeLocaleFormatter;
  exports.daysBetween = daysBetween;
  exports.daysInMonth = daysInMonth;
  exports.diffTimestamp = diffTimestamp;
  exports.findWeekday = findWeekday;
  exports.getDate = getDate;
  exports.getDateTime = getDateTime;
  exports.getDayIdentifier = getDayIdentifier;
  exports.getDayOfYear = getDayOfYear;
  exports.getDayTimeIdentifier = getDayTimeIdentifier;
  exports.getEndOfMonth = getEndOfMonth;
  exports.getEndOfWeek = getEndOfWeek;
  exports.getStartOfMonth = getStartOfMonth;
  exports.getStartOfWeek = getStartOfWeek;
  exports.getTime = getTime;
  exports.getTimeIdentifier = getTimeIdentifier;
  exports.getWeekday = getWeekday;
  exports.getWeekdaySkips = getWeekdaySkips;
  exports.getWorkWeek = getWorkWeek;
  exports.indexOf = indexOf;
  exports.isBetweenDates = isBetweenDates;
  exports.isLeapYear = isLeapYear;
  exports.isOverlappingDates = isOverlappingDates;
  exports.makeDate = makeDate;
  exports.makeDateTime = makeDateTime;
  exports.moveRelativeDays = moveRelativeDays;
  exports.nextDay = nextDay;
  exports.padNumber = padNumber;
  exports.parseDate = parseDate;
  exports.parseTime = parseTime;
  exports.parseTimestamp = parseTimestamp;
  exports.parsed = parsed;
  exports.prevDay = prevDay;
  exports.relativeDays = relativeDays;
  exports.today = today;
  exports.updateDayOfYear = updateDayOfYear;
  exports.updateDisabled = updateDisabled;
  exports.updateFormatted = updateFormatted;
  exports.updateMinutes = updateMinutes;
  exports.updateRelative = updateRelative;
  exports.updateWeekday = updateWeekday;
  exports.updateWorkWeek = updateWorkWeek;
  exports.validateNumber = validateNumber;
  exports.validateTimestamp = validateTimestamp;
  exports.version = version;
  exports.weeksBetween = weeksBetween;

  Object.defineProperty(exports, '__esModule', { value: true });

})));
