/** Express router providing user account related routes
 * @module routers/account
 * @requires express
 * @requires config - app config
 * @requires utils - app utils functions
 * @requires express-validator - form validation module
 * @requires models- app model module
 */


 /**
 * express module
 * @const
 */
const express = require('express');


/**
 * Express router to mount user page functions.
 * @type {object}
 * @const
 */
const router = express.Router();


/**
 * App utils functions module
 * @const
 */
const utils = require('../helpers/utils.js');


/**
 * App config module
 * @const
 */
const config = require('../config.js');


/**
 * Form input validation module
 * @const
 */
const { body, validationResult } = require('express-validator');
const Rbac = require('../helpers/rbac.js');


/**
 *  models
 * @const
 */
const models = require('../models/index.js');
const User = models.User;


const sequelize = models.sequelize; // sequelize functions and operations
const Op = models.Op; // sequelize query operators


/**
 * Route to view user account record
 * @route {GET} /account
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/','/index'], async (req, res) => {
	try{
		let userId = req.user.id;
		let query = {
			where: {id: userId }
		}
		query.attributes = User.accountviewFields();
		query.raw = true;
		let record = await User.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to get  user account record for edit
 * @route {GET} /account/edit
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.get('/edit', async (req, res) => {
	try{
		let userId = req.user.id;
		let query = {
			where: {id: userId }
		}
		query.attributes = User.accounteditFields();
		query.raw = true;
		let record = await User.findOne(query);
		if(!record){
			return res.badRequest("No record found");
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to update  user account record
 * @route {POST} /account/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/edit' , 
	[
		body('name').optional({nullable: true}).not().isEmpty(),
		body('foto').optional({nullable: true}).not().isEmpty(),
		body('saldo').optional({nullable: true}).not().isEmpty(),
		body('port').optional({nullable: true}).not().isEmpty().isNumeric(),
		body('user_role_id').optional().isNumeric(),
	]
, async (req, res) => {
	try{
		// Finds the validation errors in this request and wraps them in an object with handy functions
		let errors = validationResult(req);
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let recid = req.user.id;
		let modeldata = req.body;
		
        // move uploaded file from temp directory to destination directory
		if(modeldata.foto !== undefined) {
			let fileInfo = utils.moveUploadedFiles(modeldata.foto, "foto");
			modeldata.foto = fileInfo.filepath;
		}
		let query = {}
		let where =  {id: recid }
		query.where = where;
		query.attributes = User.accounteditFields();
		query.raw = true;
		let record = await User.findOne(query);
		if(!record){
			return res.notFound();
		}
		await User.update(modeldata, {where: where});
		return res.ok(modeldata);
	}
	catch(err){
		return res.serverError(err);
	}
});
router.get('/currentuserdata', async function (req, res)
{
	let user = req.user;
	let userRole = user.user_role_id;
	let rbac = new Rbac(userRole);
	let userPages = await rbac.getUserPages();
	let userRoleNames = await rbac.getRoleNames();
	let data = {
		user: user,
		pages: userPages,
		roles: userRoleNames
	}
    return res.ok(data);
});


/**
 * Route to change user password
 * @route {POST} /account
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.post('/changepassword' , 
	[
		body('oldpassword').not().isEmpty(),
		body('newpassword').not().isEmpty(),
		body('confirmpassword').not().isEmpty().custom((value, {req}) => (value === req.body.newpassword))
	]
, async function (req, res) {
	try{
		let errors = validationResult(req);
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let oldPassword = req.body.oldpassword;
		let newPassword = req.body.newpassword;
		let query = {}
		let where =  {id:  req.user.id};
		query.where = where;
		query.attributes = ['senha'];
		query.raw = true;
		let user = await User.findOne(query);
		let currentPasswordHash = user.senha;
		if(!utils.passwordVerify(oldPassword, currentPasswordHash)){
			return res.badRequest("Current password is incorrect");
		}
		let modeldata = {
			senha: utils.passwordHash(newPassword)
		}
		await User.update(modeldata, {where: where});
		return res.ok("Password change completed");
	}
	catch(err){
		return res.serverError(err);
	}
});
module.exports = router;
