/** Express router providing App_Logs related routes
 * @module routers/App_Logs
 * @requires express
 * @requires config - app config
 * @requires utils - app utils functions
 * @requires express-validator - form validation module
 * @requires models- app model module
 */


 /**
 * express module
 * @const
 */
const express = require('express');


/**
 * Express router to mount user page functions.
 * @type {object}
 * @const
 */
const router = express.Router();


/**
 * App config module
 * @const
 */
const config = require('../config.js');


/**
 * App utils functions module
 * @const
 */
const utils = require('../helpers/utils.js');


/**
 * Form input validation module
 * @const
 */
const { body, validationResult } = require('express-validator');


/**
 * App_Logs models
 * @const
 */
const models = require('../models/index.js');
const App_Logs = models.App_Logs;


const sequelize = models.sequelize; // sequelize functions and operations
const Op = models.Op; // sequelize query operators




/**
 * Route to list app_logs records
 * @route {GET} /app_logs/index/{fieldname}/{fieldvalue}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/', '/index/:fieldname?/:fieldvalue?'], async (req, res) => {  
	try{
		let query = {};  // sequelize query object
		let where = {};  // sequelize where conditions
		let replacements = {};  // sequelize query params
		let fieldname = req.params.fieldname;
		let fieldvalue = req.params.fieldvalue;
		
		if (fieldname){
			where[Op.and] = [
				sequelize.literal(`(${fieldname} = :fieldvalue)`)
			];
			replacements.fieldvalue = fieldvalue;
		}
		let search = req.query.search;
		if(search){
			let searchFields = App_Logs.searchFields();
			where[Op.or] = searchFields;
			replacements.search = `%${search}%`;
		}
		
		
		query.raw = true;
		query.where = where;
		query.replacements = replacements;
		query.order = App_Logs.getOrderBy(req);
		query.attributes = App_Logs.listFields();
		let page = parseInt(req.query.page) || 1;
		let limit = parseInt(req.query.limit) || 20;
		let result = await App_Logs.paginate(query, page, limit);
		return res.ok(result);
	}
	catch(err) {
		return res.serverError(err);
	}
});


/**
 * Route to view App_Logs record
 * @route {GET} /app_logs/view/{recid}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/view/:recid'], async (req, res) => {
	try{
		let recid = req.params.recid || null;
		let query = {}
		let where = {}
		where['log_id'] = recid;
		query.raw = true;
		query.where = where;
		query.attributes = App_Logs.viewFields();
		let record = await App_Logs.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to insert App_Logs record
 * @route {POST} /app_logs/add
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/add/' , 
	[
		body('timestamp').optional(),
		body('action').optional(),
		body('tablename').optional(),
		body('recordid').optional(),
		body('sqlquery').optional(),
		body('userid').optional(),
		body('serverip').optional(),
		body('requesturl').optional(),
		body('requestdata').optional(),
		body('requestcompleted').optional(),
		body('requestmsg').optional(),
	]
, async function (req, res) {
	try{
		let errors = validationResult(req); // get validation errors if any
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let modeldata = req.body;
		
		//save App_Logs record
		let record = await App_Logs.create(modeldata);
		//await record.reload(); //reload the record from database
		let recid =  record['log_id'];
		
		return res.ok(record);
	} catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to get  App_Logs record for edit
 * @route {GET} /app_logs/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.get('/edit/:recid', async (req, res) => {
	try{
		let recid = req.params.recid;
		let query = {};
		let where = {};
		where['log_id'] = recid;
		query.where = where;
		query.attributes = App_Logs.editFields();
		let record = await App_Logs.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to update  App_Logs record
 * @route {POST} /app_logs/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/edit/:recid' , 
	[
		body('timestamp').optional(),
		body('action').optional(),
		body('tablename').optional(),
		body('recordid').optional(),
		body('sqlquery').optional(),
		body('userid').optional(),
		body('serverip').optional(),
		body('requesturl').optional(),
		body('requestdata').optional(),
		body('requestcompleted').optional(),
		body('requestmsg').optional(),
	]
, async (req, res) => {
	try{
		let errors = validationResult(req); // get validation errors if any
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let recid = req.params.recid;
		let modeldata = req.body;
		let query = {};
		let where = {};
		where['log_id'] = recid;
		query.where = where;
		query.attributes = App_Logs.editFields();
		let record = await App_Logs.findOne(query);
		if(!record){
			return res.notFound();
		}
		await App_Logs.update(modeldata, {where: where});
		return res.ok(modeldata);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to delete App_Logs record by table primary key
 * Multi delete supported by recid separated by comma(,)
 * @route {GET} /app_logs/delete/{recid}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get('/delete/:recid', async (req, res) => {
	try{
		let recid = req.params.recid || '';
		recid = recid.split(',');
		let query = {};
		let where = {};
		where['log_id'] = recid;
		query.where = where;
		let records = await App_Logs.findAll(query);
		records.forEach(async (record) => { 
			await record.destroy();
		});
		return res.ok(recid);
	}
	catch(err){
		return res.serverError(err);
	}
});
module.exports = router;
