/** Express router providing Sms related routes
 * @module routers/Sms
 * @module routers/Contacto
 * @requires express
 * @requires config - app config
 * @requires utils - app utils functions
 * @requires express-validator - form validation module
 * @requires models- app model module
 */


 /**
 * express module
 * @const
 */
const express = require('express');


/**
 * Express router to mount user page functions.
 * @type {object}
 * @const
 */
const router = express.Router();


/**
 * App config module
 * @const
 */
const config = require('../config.js');
const axios = require('axios'); // Importa o Axios

/**
 * App utils functions module
 * @const
 */
const utils = require('../helpers/utils.js');


/**
 * Form input validation module
 * @const
 */
const { body, validationResult } = require('express-validator');


/**
 * Sms models
 * @const
 */
const models = require('../models/index.js');
const Sms = models.Sms;
const Contacto = models.Contacto;


const sequelize = models.sequelize; // sequelize functions and operations
const Op = models.Op; // sequelize query operators




/**
 * Route to list sms records
 * @route {GET} /sms/index/{fieldname}/{fieldvalue}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/', '/index/:fieldname?/:fieldvalue?'], async (req, res) => {  
	try{
		let query = {};  // sequelize query object
		let where = {};  // sequelize where conditions
		let replacements = {};  // sequelize query params
		let fieldname = req.params.fieldname;
		let fieldvalue = req.params.fieldvalue;
		
		if (fieldname){
			where[Op.and] = [
				sequelize.literal(`(${fieldname} = :fieldvalue)`)
			];
			replacements.fieldvalue = fieldvalue;
		}
		let search = req.query.search;
		if(search){
			let searchFields = Sms.searchFields();
			where[Op.or] = searchFields;
			replacements.search = `%${search}%`;
		}
		where['id_user'] = req.user.id; //filter only current records
		
		
		query.raw = true;
		query.where = where;
		query.replacements = replacements;
		query.order = Sms.getOrderBy(req);
		query.attributes = Sms.listFields();
		let page = parseInt(req.query.page) || 1;
		let limit = parseInt(req.query.limit) || 20;
		let result = await Sms.paginate(query, page, limit);
		return res.ok(result);
	}
	catch(err) {
		return res.serverError(err);
	}
});


/**
 * Route to view Sms record
 * @route {GET} /sms/view/{recid}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/view/:recid'], async (req, res) => {
	try{
		let recid = req.params.recid || null;
		let query = {}
		let where = {}
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.raw = true;
		query.where = where;
		query.attributes = Sms.viewFields();
		let record = await Sms.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to insert Sms record
 * @route {POST} /sms/add
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/add/', 
	[
		body('tipo').optional(),
		body('id_grupo').optional(),
		body('contacto').optional(),
		body('conteudo').optional(),
	],
	async function (req, res) {
		try {
			let errors = validationResult(req); // get validation errors if any
			if (!errors.isEmpty()) {
				let errorMsg = utils.formatValidationError(errors.array());
				return res.badRequest(errorMsg);
			}
			let modeldata = req.body;
			modeldata['status'] = "pendente";
			modeldata['id_user'] = req.user.id;
			let message = modeldata['conteudo'];

			let tipo = modeldata['tipo'];

			// Save SMS record
			let record = await Sms.create(modeldata);
			let recid = record['id'];

			if (recid) {
				// Buscar todos os contatos com o mesmo id_grupo
				const idGrupo = modeldata.id_grupo;
				const contacts = await Contacto.findAll({
					attributes: ['telefone'], // Selecionar apenas o campo 'telefone'
					where: { id_grupo: idGrupo } // Condição para buscar os contatos
				});

				// Rastrear números para garantir que nenhum SMS seja duplicado
				let sentNumbers = [];

				let contacto = modeldata['contacto'];

				if (tipo === "0" && contacto) {
					// Enviar SMS para um único número
					const phoneNumber = contacto;
					sendSMS(phoneNumber, message);

					// Atualizar o status para 'completo' no banco de dados
					await Sms.update({ status: 'completo' }, { where: { id: recid } });
				} else {
					// Enviar SMS para cada contato, respeitando o intervalo de 30 segundos
					let currentIndex = 0;

					// Função para enviar SMS com intervalo de 30 segundos
					const sendNextSMS = async () => {
						if (currentIndex >= contacts.length) {
							// Todos os SMSs foram enviados
							console.log("Todos os SMSs foram enviados.");
							await Sms.update({ status: 'completo' }, { where: { id: recid } });
							return;
						}

						const contact = contacts[currentIndex];
						const phoneNumber = contact.telefone;

						// Verificar se o número já foi enviado
						if (!sentNumbers.includes(phoneNumber)) {
							// Enviar o SMS
							sendSMS(phoneNumber, message);

							// Adicionar o número à lista de números enviados
							sentNumbers.push(phoneNumber);
							console.log(`SMS enviado para ${phoneNumber}. Esperando 30 segundos antes de enviar o próximo...`);
						}

						// Aguardar 30 segundos antes de enviar o próximo SMS
						currentIndex++;
						setTimeout(sendNextSMS, 30000); // 30 segundos
					};

					// Iniciar o envio de SMS
					sendNextSMS();
				}
			}

			return res.ok(record);
		} catch (err) {
			return res.serverError(err);
		}
	}
);



function sendSMS(phoneNumber, message) {
	const https = require('https');
	const querystring = require('querystring');
	const userPort = '2'; // Substitua com o valor real
	// Detalhes da API
	const url = new URL('https://sms.mozserver.com');
	const postData = querystring.stringify({
		message: message,
		destination: phoneNumber,
		account: 'apiuser', // Substitua com as credenciais reais da API
		password: 'apipass', // Substitua com as credenciais reais da API
		port: userPort
	});
	
	const options = {
		method: 'POST',
		headers: {
			'Content-Type': 'application/x-www-form-urlencoded',
			'Content-Length': postData.length
		}
	};
	
	// Enviar a requisição HTTPS
	const req = https.request(url, options, res => {
		let data = '';
	
		res.on('data', chunk => {
			data += chunk;
		});
	
		res.on('end', () => {
			console.log('Resposta da API:', data);
		});
	});
	
	req.on('error', error => {
		console.error('Erro ao enviar SMS:', error.message);
	});
	
	// Enviar os dados
	req.write(postData);
	req.end();
	
}


/**
 * Route to get  Sms record for edit
 * @route {GET} /sms/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.get('/edit/:recid', async (req, res) => {
	try{
		let recid = req.params.recid;
		let query = {};
		let where = {};
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.where = where;
		query.attributes = Sms.editFields();
		let record = await Sms.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to update  Sms record
 * @route {POST} /sms/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/edit/:recid' , 
	[
		body('tipo').optional(),
		body('id_grupo').optional(),
		body('contacto').optional(),
		body('conteudo').optional(),
	]
, async (req, res) => {
	try{
		let errors = validationResult(req); // get validation errors if any
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let recid = req.params.recid;
		let modeldata = req.body;
		let query = {};
		let where = {};
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.where = where;
		query.attributes = Sms.editFields();
		let record = await Sms.findOne(query);
		if(!record){
			return res.notFound();
		}
		await Sms.update(modeldata, {where: where});
		return res.ok(modeldata);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to delete Sms record by table primary key
 * Multi delete supported by recid separated by comma(,)
 * @route {GET} /sms/delete/{recid}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get('/delete/:recid', async (req, res) => {
	try{
		let recid = req.params.recid || '';
		recid = recid.split(',');
		let query = {};
		let where = {};
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.where = where;
		let records = await Sms.findAll(query);
		records.forEach(async (record) => { 
			await record.destroy();
		});
		return res.ok(recid);
	}
	catch(err){
		return res.serverError(err);
	}
});
async function sendMailOnRecordAdd(record){
	try{
		let mailer = require('../helpers/mailer.js');
		let ejs = require('ejs');
		let mailtitle = `New Sms Record Added`;
		let message = `New Sms record has been added`;
		let recid = record['id'];
		let baseUrl = config.app.frontendUrl;
		let recordLink = `${baseUrl}/#/sms/view/${recid}`;
		let viewData = { message, recordLink };
		let mailbody = await ejs.renderFile("views/notifications/record_action_mail.ejs", viewData);
		let recipient = `admin@sms.com`;
		let mailResult = await mailer.sendMail(recipient, mailtitle , mailbody);
		if(mailResult.messageId){
			console.log("Email Sent");
		}
		else{
			console.log(mailResult.error);
		}
	}
	catch(error){
		console.log(error)
	}
}
module.exports = router;
